package org.jboss.mx.loading;

import java.net.URL;
import java.security.CodeSource;
import java.security.Permission;
import java.security.PermissionCollection;
import java.security.Policy;
import java.security.ProtectionDomain;
import java.util.Enumeration;

import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.jboss.logging.Logger;

/**
* A port of the UnifiedClassLoader3 to a DomainClassLoader
*
* @author <a href="marc.fleury@jboss.org">Marc Fleury</a>
* @author <a href="christoph.jung@jboss.org">Christoph G. Jung</a>
* @author <a href="scott.stark@jboss.org">Scott Stark</a>
* @author <a href="juha@jboss.org">Juha Lindfors</a>
* @author <a href="bill@jboss.org">Bill Burke</a>
* @version $Revision: 44243 $
*/
public class DomainClassLoaderUCLImpl extends LegacyDomainClassLoader
   implements UnifiedClassLoaderMBean
{
   // Static --------------------------------------------------------

   private static final Logger log = Logger.getLogger(DomainClassLoaderUCLImpl.class);

   // Attributes ----------------------------------------------------

   // Constructors --------------------------------------------------
   /**
    * Construct a DomainClassLoaderUCLImpl with the given classpath and register
    * it to the given repository.
    * @param cp - the loader classpath
    * @param repository - the repository this classloader delegates to
    */
   public DomainClassLoaderUCLImpl(URL[] cp, LoaderRepositoryDomain repository)
   {
      super(cp, null);

      // register this loader to the given repository
      repository.addClassLoader(this);
   }

   // Public --------------------------------------------------------

   /** Obtain the ObjectName under which the UCL can be registered with the
    JMX server. This creates a name of the form "jmx.loading:UCL=hashCode"
    since we don't currently care that UCL be easily queriable.
    */
   public ObjectName getObjectName() throws MalformedObjectNameException
   {
      String name = "jmx.loading:UCL="+Integer.toHexString(super.hashCode());
      return new ObjectName(name);
   }

   /**
    * Get the class loader package names from the class loader URLs
    */
   public String[] getPackageNames()
   {
      UnifiedLoaderRepositoryDCL ulr = (UnifiedLoaderRepositoryDCL) domain;
      return ulr.getPackageNames(this);
   }

   public void unregister()
   {
      super.unregister();
   }

   public synchronized Class loadClassImpl(String name, boolean resolve, int stopAt)
      throws ClassNotFoundException
   {
      loadClassDepth ++;
      boolean trace = log.isTraceEnabled();

      if( trace )
         log.trace("loadClassImpl, name="+name+", resolve="+resolve);
      if( domain == null )
      {
         // If we have been undeployed we can still try locally
         try
         {
            return super.loadClass(name, resolve);
         }
         catch (ClassNotFoundException ignored)
         {
         }
         String msg = "Invalid use of destroyed classloader, UCL destroyed at:";
         throw new ClassNotFoundException(msg, this.unregisterTrace);
      }

      /* Since loadClass can be called from loadClassInternal with the monitor
         already held, we need to determine if there is a ClassLoadingTask
         which requires this UCL. If there is, we release the UCL monitor
         so that the ClassLoadingTask can use the UCL.
       */
      boolean acquired = attempt(1);
      while( acquired == false )
      {
         /* Another thread needs this UCL to load a class so release the
          monitor acquired by the synchronized method. We loop until
          we can acquire the class loading lock.
         */
        try
         {
            if( trace )
               log.trace("Waiting for loadClass lock");
            this.wait();
         }
         catch(InterruptedException ignore)
         {
         }
         acquired = attempt(1);
      }

      ClassLoadingTaskDCL task = null;
      try
      {
         Thread t = Thread.currentThread();
         // Register this thread as owning this UCL
         if( loadLock.holds() == 1 )
            LoadMgrDCL.registerLoaderThread(this, t);

         // Create a class loading task and submit it to the repository
         task = new ClassLoadingTaskDCL(name, this, t, stopAt);
         /* Process class loading tasks needing this UCL until our task has
            been completed by the thread owning the required UCL(s).
          */
         UnifiedLoaderRepositoryDCL repository = (UnifiedLoaderRepositoryDCL) domain;
         if( LoadMgrDCL.beginLoadTask(task, repository) == false )
         {
            while( task.threadTaskCount != 0 )
            {
               try
               {
                  LoadMgrDCL.nextTask(t, task, repository);
               }
               catch(InterruptedException e)
               {
                  // Abort the load or retry?
                  break;
               }
            }
         }
      }
      finally
      {
         // Unregister as the UCL owner to reschedule any remaining load tasks
         if( loadLock.holds() == 1 )
            LoadMgrDCL.endLoadTask(task);
         // Notify any threads waiting to use this UCL
         this.release();
         this.notifyAll();
         loadClassDepth --;
      }

      if( task.loadedClass == null )
      {
         if( task.loadException instanceof ClassNotFoundException )
            throw (ClassNotFoundException) task.loadException;
         else if( task.loadException instanceof NoClassDefFoundError )
            throw (NoClassDefFoundError) task.loadException;
         else if( task.loadException != null )
         {
            if( log.isTraceEnabled() )
               log.trace("Unexpected error during load of:"+name, task.loadException);
            String msg = "Unexpected error during load of: "+name
               + ", msg="+task.loadException.getMessage();
            ClassNotFoundException cnfe = new ClassNotFoundException(msg, task.loadException);
            throw cnfe;
         }
         // Assert that loadedClass is not null
         else
            throw new IllegalStateException("ClassLoadingTask.loadedTask is null, name: "+name);
      }

      return task.loadedClass;
   }

   // URLClassLoader overrides --------------------------------------

   /** Override the permissions accessor to use the CodeSource
    based on the original URL if one exists. This allows the
    security policy to be defined in terms of the static URL
    namespace rather than the local copy or nested URL.
    This builds a PermissionCollection from:
    1. The origURL CodeSource
    2. The argument CodeSource
    3. The Policy.getPermission(origURL CodeSource)

    This is necessary because we cannot define the CodeSource the
    SecureClassLoader uses to register the class under.

    @param cs the location and signatures of the codebase.
    */
   protected PermissionCollection getPermissions(CodeSource cs)
   {
      CodeSource permCS = cs;
      Policy policy = Policy.getPolicy();
      PermissionCollection perms = super.getPermissions(permCS);
      PermissionCollection perms2 = super.getPermissions(cs);
      PermissionCollection perms3 = policy.getPermissions(permCS);
      Enumeration iter = perms2.elements();
      while( iter.hasMoreElements() )
         perms.add((Permission) iter.nextElement());
      iter = perms3.elements();
      while( iter.hasMoreElements() )
         perms.add((Permission) iter.nextElement());
      if( log.isTraceEnabled() )
         log.trace("getPermissions, cp: "+getURLs()+" -> "+perms);
      return perms;
   }

}
