/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.management.builders;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.plugins.ManagedObjectImpl;
import org.jboss.managed.plugins.ManagedPropertyImpl;
import org.jboss.metatype.api.types.SimpleMetaType;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValueSupport;
import org.jboss.util.xml.DOMWriter;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * A ManagedObject impl that manages a jaxp Document attachment.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision:$
 */
public class DomDataSourceManagedObject extends ManagedObjectImpl
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 1L;
   private static String[] dsTypes = {"local-tx-datasource", "no-tx-datasource", "xa-datasource"};

   /** The document */
   private Document document;

   /**
    * DomDataSourceManagedObject ctor
    */
   public DomDataSourceManagedObject(String attachmentName, Document document)
   {
      super(attachmentName);
      this.document = document;
      Element element;
      if( document == null )
      {
         try
         {
            // Create a skeleton datasources/local-tx-datasource
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            DOMImplementation impl = builder.getDOMImplementation();
            document = impl.createDocument(null, null, null);
            
            element = document.createElement("datasources");
            document.appendChild(element);
            // Create a local-tx-datasource
            Element child = document.createElement("local-tx-datasource");
            element.appendChild(child);
            element = child;
         }
         catch (RuntimeException e)
         {
            throw e;
         }
         catch (Exception e)
         {
            throw new RuntimeException("Error creating dom", e);
         }
      }
      else
      {
         NodeList dses = document.getElementsByTagName("datasources");
         if( dses.getLength() == 0 )
         {
            element = document.createElement("datasources");
            document.appendChild(element);
         }
         else
         {
            element = (Element) dses.item(0);
         }
         NodeList dsType = null;
         for(String type : dsTypes)
         {
            dsType = element.getElementsByTagName(type);
            if( dsType.getLength() > 0 )
               break;
         }

         if( dsType == null || dsType.getLength() == 0 )
         {
            // Create a local-tx-datasource
            Element child = document.createElement("local-tx-datasource");
            element.appendChild(child);
            element = child;
         }
         else
         {
            element = (Element) dsType.item(0);
         }
      }

      Collection<ManagedProperty> properties = getProperties().values();
      // The datasource-type, TODO this needs different handling since this maps to a element name, not a value
      DomFields dsType = new DomFields(element, "datasource-type");
      dsType.setField(Fields.DESCRIPTION, "The type of DataSource");
      dsType.setField(Fields.MANDATORY, Boolean.TRUE);
      dsType.setField(Fields.META_TYPE, SimpleMetaType.STRING);
      HashSet<MetaValue> values = new HashSet<MetaValue>();
      values.add(SimpleValueSupport.wrap("local-tx-datasource"));
      values.add(SimpleValueSupport.wrap("no-tx-datasource"));
      values.add(SimpleValueSupport.wrap("xa-datasource"));
      dsType.setLegalValues(values);

      // The jndi name field info
      DomFields jndiName = new DomFields(element, "jndi-name");
      jndiName.setField(Fields.DESCRIPTION, "The jndi name to bind the DataSource under");
      jndiName.setField(Fields.MANDATORY, Boolean.TRUE);
      jndiName.setField(Fields.META_TYPE, SimpleMetaType.STRING);
      // The driver-class field info
      DomFields driverClass = new DomFields(element, "driver-class");
      driverClass.setField(Fields.DESCRIPTION, "The jdbc driver class DataSource");
      driverClass.setField(Fields.MANDATORY, Boolean.TRUE);
      driverClass.setField(Fields.META_TYPE, SimpleMetaType.STRING);
      // The connection-url field info
      DomFields connURL = new DomFields(element, "connection-url");
      connURL.setField(Fields.DESCRIPTION, "The jdbc url of the DataSource");
      connURL.setField(Fields.MANDATORY, Boolean.TRUE);
      connURL.setField(Fields.META_TYPE, SimpleMetaType.STRING);
      // The user field info
      DomFields user = new DomFields(element, "user-name");
      user.setField(Fields.DESCRIPTION, "The username for the connection-url");
      user.setField(Fields.MANDATORY, Boolean.FALSE);
      user.setField(Fields.META_TYPE, SimpleMetaType.STRING);
      // The user field info
      DomFields password = new DomFields(element, "password", false);
      password.setField(Fields.DESCRIPTION, "The password for the connection-url");
      password.setField(Fields.MANDATORY, Boolean.FALSE);
      password.setField(Fields.META_TYPE, SimpleMetaType.STRING);
      // The min-pool-size field info
      DomFields minPoolSize = new DomFields(element, "min-pool-size");
      password.setField(Fields.DESCRIPTION, "The min pool size");
      password.setField(Fields.MANDATORY, Boolean.FALSE);
      password.setField(Fields.META_TYPE, SimpleMetaType.INTEGER);
      // The max-pool-size field info
      DomFields maxPoolSize = new DomFields(element, "max-pool-size");
      password.setField(Fields.DESCRIPTION, "The max pool size");
      password.setField(Fields.MANDATORY, Boolean.FALSE);
      password.setField(Fields.META_TYPE, SimpleMetaType.INTEGER);

      // The security-domain field info
      DomFields secDomain = new DomFields(element, "security-domain");
      secDomain.setField(Fields.DESCRIPTION, "The security-domain used to validate connections");
      secDomain.setField(Fields.MANDATORY, Boolean.FALSE);
      secDomain.setField(Fields.META_TYPE, SimpleMetaType.STRING);
      // prepared-statement-cache-size
      DomFields psCacheSize = new DomFields(element, "prepared-statement-cache-size");
      secDomain.setField(Fields.DESCRIPTION, "The prepared statement cache size");
      secDomain.setField(Fields.MANDATORY, Boolean.FALSE);
      secDomain.setField(Fields.META_TYPE, SimpleMetaType.INTEGER);
      

      properties.add(new ManagedPropertyImpl(this, dsType));
      properties.add(new ManagedPropertyImpl(this, jndiName));
      properties.add(new ManagedPropertyImpl(this, driverClass));
      properties.add(new ManagedPropertyImpl(this, connURL));
      properties.add(new ManagedPropertyImpl(this, user));
      properties.add(new ManagedPropertyImpl(this, password));
      properties.add(new ManagedPropertyImpl(this, minPoolSize));
      properties.add(new ManagedPropertyImpl(this, maxPoolSize));
      properties.add(new ManagedPropertyImpl(this, secDomain));
      properties.add(new ManagedPropertyImpl(this, psCacheSize));
      
   }

   public String prettyPrint()
   {
      StringBuffer tmp = new StringBuffer(super.toString());
      tmp.append('[');
      tmp.append(DOMWriter.printNode(document, true));
      tmp.append(']');
      return tmp.toString();
   }

   @Override
   public String toString()
   {
      return prettyPrint();
   }
   
}
