/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.management.templates;

import java.io.File;
import java.io.FileWriter;
import java.net.URI;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.jboss.deployers.spi.management.DeploymentTemplate;
import org.jboss.deployers.spi.management.DeploymentTemplateInfo;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.metatype.api.types.CompositeMetaType;
import org.jboss.metatype.api.values.CompositeValue;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.util.xml.DOMWriter;
import org.jboss.virtual.VirtualFile;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * A template for creating a datasource(*-ds.xml) deployments.
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision:$
 */
public class DsXmlDataSourceTemplate
   implements DeploymentTemplate
{
   private static final Logger log = Logger.getLogger(DsXmlDataSourceTemplate.class);
   private DeploymentTemplateInfo info;

   /**
    * Creates a root/{deploymentBaseName}-ds.xml base descriptor.
    */
   public VirtualFile applyTemplate(VirtualFile root, String deploymentBaseName,
         DeploymentTemplateInfo values)
      throws Exception
   {
      String dsName = deploymentBaseName+"-ds.xml";
      URI dsXmlURI = new URI(root.toURI()+dsName);
      File dsXml = new File(dsXmlURI.getPath());
      writeTemplate(dsXml, values);
      return root.findChild(dsName);
   }

   /**
    * There is no update of the ctx by this template since the data model
    * is driven from the descriptor created by applyTemplate.
    * 
    * @param ctx - the DeploymentContext to associate metadata to
    * @param values - the managed property values passed to apply.
    */
   public void updateTemplateDeployment(VFSDeployment ctx,
         DeploymentTemplateInfo values)
      throws Exception
   {
   }

   public DeploymentTemplateInfo getInfo()
   {
      return info;
   }
   public void setInfo(DeploymentTemplateInfo info)
   {
      this.info = info;
   }

   protected void writeTemplate(File dsXml, DeploymentTemplateInfo values)
      throws Exception
   {
      FileWriter fw = new FileWriter(dsXml);
      fw.write("<datasources>\n");
      Map<String, ManagedProperty> properties = values.getProperties();
      // Get the datasource-type
      ManagedProperty dsType = properties.get("datasource-type");
      if( dsType == null || dsType.getValue() == null )
         throw new IllegalStateException("Required datasource-type value not found");
      fw.write('<');
      fw.write(dsType.getValue().toString());
      fw.write(">\n");

      // All other properties are simple text child elements
      for(ManagedProperty p : properties.values())
      {
         String tagName = p.getName();
         if( tagName.equals("datasource-type") )
            continue;
         if( p.isMandatory() && p.getValue() == null )
            throw new IllegalStateException("Required "+tagName+" value not found");
         fw.write("  <");
         fw.write(tagName);
         fw.write('>');
         // Except for connection-properties
         if( p.getMetaType() instanceof CompositeMetaType )
         {
            CompositeValue cvalue = (CompositeValue) p.getValue();
            for(String key : cvalue.getMetaType().keySet())
            {
               if( cvalue.containsKey(key) )
               {
                  SimpleValue svalue = (SimpleValue) cvalue.get(key);
                  String string = svalue.toString();
                  fw.write("  <");
                  fw.write(key);
                  fw.write('>');
                  fw.write(string);
                  fw.write("</");
                  fw.write(tagName);
                  fw.write(">\n");                  
               }
            }
         }
         else
         {
            Object value = p.getValue();
            if( value == null )
               value = "";
            fw.write(value.toString());
            fw.write("</");
            fw.write(tagName);
            fw.write(">\n");
         }
      }
      // close dsType tag
      fw.write("</");
      fw.write(dsType.getValue().toString());
      fw.write(">\n");
      fw.write("</datasources>\n");
      fw.flush();
      fw.close();
   }

   protected Document buildDocument(DeploymentTemplateInfo values)
      throws Exception
   {
      Document document = null;
      DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      DocumentBuilder builder = factory.newDocumentBuilder();
      DOMImplementation impl = builder.getDOMImplementation();
      document = impl.createDocument(null, null, null);
      Element ds = document.createElement("datasources");
      document.appendChild(ds);
      Map<String, ManagedProperty> properties = values.getProperties();
      // datasource-type
      ManagedProperty dsType = properties.get("datasource-type");
      if( dsType == null || dsType.getValue() == null )
         throw new IllegalStateException("Required datasource-type value not found");
      Element dsTypeElem = document.createElement(dsType.getValue().toString());
      ds.appendChild(dsTypeElem);
      // All other
      for(ManagedProperty p : properties.values())
      {
         String tagName = p.getName();
         if( tagName.equals("datasource-type") )
            continue;
         if( p.isMandatory() && p.getValue() == null )
            throw new IllegalStateException("Required "+tagName+" value not found");
         Element element = document.createElement(tagName);
         Text t = document.createTextNode(p.getValue().toString());
         element.appendChild(t);
         dsTypeElem.appendChild(element);
      }

      log.debug("Updated metadata to: "+DOMWriter.printNode(document, true));
      return document;
   }

}
