/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.management.templates;

import java.io.File;
import java.io.FileWriter;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.Marshaller;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlValue;
import org.jboss.deployers.spi.management.DeploymentTemplate;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.managed.api.DeploymentTemplateInfo;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.virtual.VirtualFile;

/**
 * A template for creating jms destinations
 * 
 * @author <a href="mailto:alex@jboss.org">Alexey Loubyansky</a>
 * @version <tt>$Revision: $</tt>
 */
public class JmsDestinationTemplate
   implements DeploymentTemplate
{
   private DeploymentTemplateInfo info;

   public DeploymentTemplateInfo getInfo()
   {
      return info;
   }

   public void setInfo(DeploymentTemplateInfo info)
   {
      this.info = info;
   }

   public VirtualFile applyTemplate(VirtualFile root, String deploymentBaseName, DeploymentTemplateInfo values)
      throws Exception
   {
      String dsName = deploymentBaseName + "-service.xml";
      URI dsXmlURI = new URI(root.toURI() + dsName);
      File dsXml = new File(dsXmlURI.getPath());
      writeTemplate(dsXml, values);
      return root.findChild(dsName);
   }

   public void updateTemplateDeployment(VFSDeployment ctx, DeploymentTemplateInfo values) throws Exception
   {
   }

   private void writeTemplate(File dsXml, DeploymentTemplateInfo info)
      throws Exception
   {
      JmsDestinationTemplateInfo destinationInfo = (JmsDestinationTemplateInfo)info;
      String destinationType = destinationInfo.getDestinationType();

      JmsDestinationMetaData destination = new JmsDestinationMetaData();
      String jndiName = "" + getProperty(info, "JNDIName");
      if("queue".equals(destinationType))
      {
         destination.className = "org.jboss.jms.server.destination.QueueService";
         destination.annotation =
            "@org.jboss.system.deployers.managed.ManagementObjectClass(code=org.jboss.jms.server.destination.QueueServiceMO)";
         destination.xmbeanDd = "xmdesc/Queue-xmbean.xml";
         destination.jmxName = "jboss.messaging.destination:service=Queue,name=" + jndiName;
      }
      else if("topic".equals(destinationType))
      {
         destination.className = "org.jboss.jms.server.destination.TopicService";
         destination.annotation =
            "@org.jboss.system.deployers.managed.ManagementObjectClass(code=org.jboss.jms.server.destination.TopicServiceMO)";
         destination.xmbeanDd = "xmdesc/Topic-xmbean.xml";
         destination.jmxName = "jboss.messaging.destination:service=Topic,name=" + jndiName;
      }
      else
      {
         throw new IllegalStateException("Unsupported destination type: " + destinationType);
      }
      List<JmsAttributeMetaData> attributes = new ArrayList<JmsAttributeMetaData>();

      JmsAttributeMetaData jndiNameMD = new JmsAttributeMetaData("JNDIName", jndiName);
      destination.attribute = attributes;
      attributes.add(jndiNameMD);

      List<JmsDependencyMetaData> depends = new ArrayList<JmsDependencyMetaData>();
      destination.depends = depends;
      depends.add(new JmsDependencyMetaData("ServerPeer", "jboss.messaging:service=ServerPeer"));
      depends.add(new JmsDependencyMetaData("jboss.messaging:service=PostOffice"));

      JmsDestinationDeployment deployment = new JmsDestinationDeployment();
      deployment.destination = destination;

      Class[] classes = {JmsDestinationDeployment.class};
      JAXBContext context = JAXBContext.newInstance(classes);
      Marshaller marshaller = context.createMarshaller();
      marshaller.setProperty("jaxb.formatted.output", Boolean.TRUE);

      JAXBElement<JmsDestinationDeployment> root =
         new JAXBElement<JmsDestinationDeployment>(
            new javax.xml.namespace.QName("server"),
            JmsDestinationDeployment.class,
            null, deployment
         );

      FileWriter fw = null;
      try
      {
         fw = new FileWriter(dsXml);
         marshaller.marshal(root, fw);
      }
      finally
      {
         if(fw != null)
         {
            fw.close();
         }
      }
   }

   /**
    * Extract the value from the property MetaValue
    * @param info - template info
    * @param propName - the name of the property to return a value for
    * @return the unwrapped property value
    */
   private Object getProperty(DeploymentTemplateInfo info, String propName)
   {
      Map<String, ManagedProperty> propsInfo = info.getProperties();
      ManagedProperty prop = propsInfo.get(propName);
      if(prop == null)
      {
         throw new IllegalStateException("Property " + propName + " not found.");
      }
      Object value = prop.getValue();
      if(value instanceof MetaValue)
      {
         if (prop.getMetaType().isComposite())
         {
            throw new IllegalArgumentException("Unable to handle composite: "+prop);
         }
         else if(prop.getMetaType().isSimple())
         {
            SimpleValue sv = SimpleValue.class.cast(value);
            value = sv.getValue();
         }
      }
      return value;
   }

   // the classes below should go away and ServiceDeployment and MetaData
   // API should be used instead once their bound with JAXB

   @XmlRootElement(name = "server")
   public static class JmsDestinationDeployment
   {
      @XmlElement(name = "mbean")
      public JmsDestinationMetaData destination;
   }

   public static class JmsDestinationMetaData
   {
      @XmlAttribute(name = "code")
      String className;

      @XmlAttribute(name = "name")
      String jmxName;

      @XmlAttribute(name = "xmbean-dd")
      String xmbeanDd;

      @XmlElement
      String annotation;

      public List<JmsAttributeMetaData> attribute;
      public List<JmsDependencyMetaData> depends;
   }

   public static class JmsAttributeMetaData
   {
      @XmlAttribute
      String name;
      @XmlValue
      String value;
      public JmsAttributeMetaData()
      {
      }
      public JmsAttributeMetaData(String name, String value)
      {
         this.name = name;
         this.value = value;
      }
   }
   public static class JmsDependencyMetaData
   {
      @XmlAttribute(name = "optional-attribute-name")
      String attribute;

      @XmlValue
      String value;

      public JmsDependencyMetaData()
      {
      }


      public JmsDependencyMetaData(String value)
      {
         this.value = value;
      }


      public JmsDependencyMetaData(String attribute, String value)
      {
         this.attribute = attribute;
         this.value = value;
      }
   }
}
