/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.deployers.managed;

import java.io.Serializable;
import java.util.List;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.jboss.beans.info.spi.BeanInfo;
import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.spi.factory.InstanceClassFactory;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.system.metadata.ServiceAnnotationMetaData;
import org.jboss.system.metadata.ServiceAttributeMetaData;
import org.jboss.system.metadata.ServiceDependencyValueMetaData;
import org.jboss.system.metadata.ServiceMetaData;
import org.jboss.system.metadata.ServiceTextValueMetaData;
import org.jboss.system.metadata.ServiceValueMetaData;

/**
 * The InstanceClassFactory implementation for ServiceMetaData.
 * 
 * @author Scott.Stark@jboss.org
 * @author Dimitris.Andreadis@jboss.org
 * @version $Revision: 69625 $
 */
public class ServiceMetaDataICF implements InstanceClassFactory<ServiceMetaData>
{
   private static final Logger log = Logger.getLogger(ServiceMetaDataICF.class);

   private static final String MOCLASS_ANNOTATION = '@' + ManagementObjectClass.class.getName();
   
   private MBeanServer mbeanServer;
   
   /** The meta value factory */
   private MetaValueFactory metaValueFactory = MetaValueFactory.getInstance(); 

   public MBeanServer getMbeanServer()
   {
      return mbeanServer;
   }

   public void setMbeanServer(MBeanServer mbeanServer)
   {
      this.mbeanServer = mbeanServer;
   }

   public Class<? extends Serializable> getManagedObjectClass(ServiceMetaData md)
      throws ClassNotFoundException
   {
      try
      {
         // Fallback to TCL if there is no mbeanServer
         ClassLoader loader = null;
         if(mbeanServer != null)
            loader = mbeanServer.getClassLoader(md.getClassLoaderName());
         if(loader == null)
            loader = Thread.currentThread().getContextClassLoader();
         Class moClass = loader.loadClass(md.getCode());

         // Looks for a ManagementObjectClass annotation that defines
         // an alternate class to scan for management annotations
         List<ServiceAnnotationMetaData> samlist = md.getAnnotations();
         for (ServiceAnnotationMetaData sam : samlist)
         {
            // Annotations are not yet introduced to the actual mbean
            // so just look for the annotation string
            String anString = sam.getAnnotation();
            if (anString.startsWith(MOCLASS_ANNOTATION))
            {
               Class<?> originalClass = moClass;
               ManagementObjectClass moc = (ManagementObjectClass)sam.getAnnotationInstance(loader);
               moClass = moc.code();
               log.debug("Using alternate class '" + moClass + "' for class " + originalClass);
               break;
            }  
         }
         return moClass;
      }
      catch(InstanceNotFoundException e)
      {
         throw new ClassNotFoundException("Failed to obtain mbean class loader", e);
      }
   }

   public MetaValue getValue(BeanInfo beanInfo, ManagedProperty property, ServiceMetaData md)
   {
      // First look to the mapped name
      String name = property.getMappedName();
      if (name == null)
         property.getName();

      // Get the attribute value
      Object value = null;
      for (ServiceAttributeMetaData amd : md.getAttributes())
      {
         // The compare is case-insensitve due to the attribute/javabean case mismatch
         if (amd.getName().equalsIgnoreCase(name))
         {
            value = amd.getValue();
            break;
         }
      }
      // Unwrap the ServiceValueMetaData types
      if (value instanceof ServiceTextValueMetaData)
      {
         ServiceTextValueMetaData text = (ServiceTextValueMetaData) value;
         value = text.getText();
      }
      else if (value instanceof ServiceDependencyValueMetaData)
      {
         ServiceDependencyValueMetaData depends = (ServiceDependencyValueMetaData) value;
         value = depends.getDependency();
      }
      // TODO: unwrap other ServiceValueMetaData types

      PropertyInfo propertyInfo = beanInfo.getProperty(name);
      MetaValue mvalue;
      try
      {
         mvalue = metaValueFactory.create(value, propertyInfo.getType());
      }
      catch(Exception e)
      {
         log.debug("Failed to get property value for bean: "+beanInfo.getName()
               +", property: "+propertyInfo.getName(), e);
         mvalue = metaValueFactory.create(null, propertyInfo.getType());
      }
      return mvalue;
   }

   public void setValue(BeanInfo beanInfo, ManagedProperty property, ServiceMetaData md, MetaValue value)
   {
      // First look to the mapped name
      String name = property.getMappedName();
      if (name == null)
         property.getName();

      // Get the attribute value
      ServiceValueMetaData attributeValue = null;
      for (ServiceAttributeMetaData amd : md.getAttributes())
      {
         // The compare is case-insensitve due to the attribute/javabean case mismatch
         if (amd.getName().equalsIgnoreCase(name))
         {
            attributeValue = amd.getValue();
            break;
         }
      }
      if (attributeValue != null)
      {
         PropertyInfo propertyInfo = beanInfo.getProperty(name);
         Object plainValue = metaValueFactory.unwrap(value, propertyInfo.getType());

         // Unwrap the ServiceValueMetaData types
         if (attributeValue instanceof ServiceTextValueMetaData)
         {
            ServiceTextValueMetaData text = (ServiceTextValueMetaData) attributeValue;
            text.setText(String.valueOf(plainValue));
         }
         else if (value instanceof ServiceDependencyValueMetaData)
         {
            ServiceDependencyValueMetaData depends = (ServiceDependencyValueMetaData) attributeValue;
            depends.setDependency(String.valueOf(plainValue));
         }
         // TODO: unwrap other ServiceValueMetaData types
      }
      else
         throw new IllegalArgumentException("No matching attribute found: " + name + "/" + md);
   }

   public Object getComponentName(BeanInfo beanInfo, ManagedProperty property, ServiceMetaData md, MetaValue value)
   {
      if (beanInfo == null || property == null || value == null)
      {
         ObjectName objectName = md.getObjectName();
         if (objectName != null)
            return objectName.getCanonicalName();
      }
      return null;
   }
}
