/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */

package org.jboss.deployers.spi.management;

import java.util.Set;

import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.ProfileKey;

/**
 * The management view plugin spi for querying profiles for the
 * deployemnt management object interface roots.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision$
 */
public interface ManagementView
{
   /**
    * Load and associate the given profile with the ManagementView
    * for future operations.
    * 
    * @param key - the profile to load
    * @throws NoSuchProfileException
    */
   public void loadProfile(ProfileKey key)
      throws Exception;

   /**
    * Get the names of the deployment in the profile.
    * 
    * @return the names
    */
   public Set<String> getDeploymentNames();

   /**
    * Get the names of the deployment in the profile that have the
    * given deployment type.
    * @param type - the deployment type
    * @return the names
    */
   public Set<String> getDeploymentNamesForType(String type);

   /**
    * Search for a deployment matching the regex expression.
    * @param regex - the regex to query deployment name 
    * @return the deployment name.
    * @throws NoSuchDeploymentException if no matches are found 
    */
   public Set<String> getMatchingDeploymentName(String regex)
      throws NoSuchDeploymentException;

   /**
    * Get a deployment
    * 
    * @param name the name
    * @param phase the phase
    * @return the deployment
    * @throws NoSuchDeploymentException if no matches are found 
    * @throws Exception for any error
    */
   public ManagedDeployment getDeployment(String name, DeploymentPhase phase)
      throws NoSuchDeploymentException, Exception;

   /**
    * Get the deployments of a type.
    * @param type - the deployment or module type.
    * @return the possibly empty set of deployment with the given type.
    * @throws Exception
    */
   public Set<ManagedDeployment> getDeploymentsForType(String type)
      throws Exception;

   /**
    * Get the components of a type. The 
    * @param type - the component type.
    * @return the possibly empty set of components with the given type.
    * @throws Exception
    */
   public Set<ManagedComponent> getComponentsForType(ComponentType type)
      throws Exception;

   /**
    * Obtain the ManagedComponent for the given name/type
    * @return the possibly null ManagedComponent
    * @throws Exception
    */
   public ManagedComponent getComponent(String name, ComponentType type)
      throws Exception;

   /**
    * Update the given component. This takes the component ManagedDeployment,
    * obtains the ManagedObject set for the underlying deployment, and applies
    * the component ManagedProperty set to matching the deployment ManagedObject
    * properties.
    * 
    * @param comp
    * @throws Exception
    */
   public void updateComponent(ManagedComponent comp)
      throws Exception;

   /**
    * Get the registered DeploymentTemplate names.
    * 
    * TODO: probably needs a Map<String, DeploymentType> notion
    * @return the template names
    */
   public Set<String> getTemplateNames();

   /**
    * Get a deployment template.
    * 
    * @param name - the deployment name to identify the template to retrieve
    * @return the named DeploymentTemplate
    * @throws NoSuchDeploymentException - if there is no such template
    */
   public DeploymentTemplateInfo getTemplate(String name)
      throws NoSuchDeploymentException;

   /**
    * 
    * @param phase
    * @param deploymentBaseName
    * @param info
    * @throws Exception
    */
   public void applyTemplate(DeploymentPhase phase,
         String deploymentBaseName, DeploymentTemplateInfo info)
         throws Exception;

   /**
    * Remove a deployment
    * 
    * @param deploymentName the deployment name
    * @param phase the phase
    * @throws NoSuchProfileException
    * @throws NoSuchDeploymentException
    * @throws Exception for any error
    */
   public void removeDeployment(String deploymentName, DeploymentPhase phase)
      throws NoSuchDeploymentException, Exception;

   /**
    * Process the changes made to the profile.
    * 
    * @throws Exception
    */
   public void process() throws Exception;
}
