/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.aop;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.aop.joinpoint.Invocation;
import org.jboss.aop.joinpoint.MethodInvocation;
import org.jboss.aop.proxy.container.AOPProxyFactoryParameters;
import org.jboss.aop.proxy.container.GeneratedAOPProxyFactory;
import org.jboss.deployers.spi.attachments.Attachments;
import org.jboss.logging.Logger;

/**
 * Duplicate aspect for testing attachments interception.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision:$
 */
public class TrackingAdvice
{
   private static Logger log = Logger.getLogger(TrackingAdvice.class);
   private static ConcurrentHashMap<Object, Map<String, Object>> attachmentsByTarget =
      new ConcurrentHashMap<Object, Map<String, Object>>();
   private static boolean trace = true;

   public static Attachments wrapAttachments(Attachments attachments)
   {
      return createProxy(attachments, Attachments.class);
   }

   public Object invoke(Invocation inv)
      throws Throwable
   {
      if( trace )
         log.trace("invoke, "+inv);
      return inv.invokeNext();
   }

   public Object addAttachment(MethodInvocation mi)
      throws Throwable
   {
      if( trace )
         log.trace(mi);
      Object target = mi.getTargetObject();
      Object[] args = mi.getArguments();
      Object value = mi.invokeNext();
      String name;
      Object attachment;
      // addAttachment(Class<T> type, T attachment)
      if( args[0] instanceof Class )
      {
         Class c = Class.class.cast(args[0]);
         name = c.getName();
         attachment = args[1];
      }
      // addAttachment(String name, T attachment, Class<T> expectedType)
      // addAttachment(String name, T attachment)
      else
      {
         name = String.class.cast(args[0]);
         attachment = args[1];
      }
      addAttachment(target, name, attachment);
      return value;
   }

   public static Object removeAttachment(MethodInvocation mi)
      throws Throwable
   {
      Object target = mi.getTargetObject();
      Object[] args = mi.getArguments();
      Object value = mi.invokeNext();
      String name;
      // removeAttachment(Class<T> type)
      if( args[0] instanceof Class )
      {
         Class c = Class.class.cast(args[0]);
         name = c.getName();
      }
      // removeAttachment(String name, Class<T> expectedType)
      // removeAttachment(String name)
      else
      {
         name = String.class.cast(args[0]);
      }
      removeAttachment(target, name);
      return value;
   }

   public static Map<String, Object> getAttachmentsForTarget(Object key)
   {
      Map<String, Object> attachments = attachmentsByTarget.get(key);
      return attachments;
   }
   public static Map<String, Object> clearAttachmentsForTarget(Object key)
   {
      Map<String, Object> attachments = attachmentsByTarget.remove(key);
      if( trace )
         log.trace("clearAttachmentsForTarget, target="+key+" :"+attachments);
      return attachments;
   }

   private static void addAttachment(Object target, String name, Object attachment)
   {
      Map<String, Object> attachments = attachmentsByTarget.get(target);
      if( attachments == null )
      {
         attachments = new HashMap<String, Object>();
         attachmentsByTarget.put(target, attachments);
      }
      attachments.put(name, attachment);
      if( trace )
         log.trace("addAttachment, target="+target+", name="+name+", attachment="+attachment);
   }
   private static void removeAttachment(Object target, String name)
   {
      Map<String, Object> attachments = attachmentsByTarget.get(target);
      if( attachments != null )
      {
         attachments.remove(name);
         if( attachments.size() == 0 )
            attachmentsByTarget.remove(target);
      }
      if( trace )
         log.trace("removeAttachment, target="+target+", name="+name);
   }

   /**
    * Create a proxy 
    * 
    * @param <T> the expected type
    * @param target the target
    * @param interfaceClass the interface class
    * @return the proxy
    */
   public static <T> T createProxy(T target, Class<T> interfaceClass)
   {
      if (target == null)
         return null;

      GeneratedAOPProxyFactory proxyFactory = new GeneratedAOPProxyFactory();
      AOPProxyFactoryParameters params = new AOPProxyFactoryParameters();
      params.setInterfaces(new Class[] { interfaceClass });
      params.setObjectAsSuperClass(true);
      params.setTarget(target);
      Object proxy = proxyFactory.createAdvisedProxy(params);
      if( trace )
         log.trace("Created proxy: "+proxy.getClass()+"@"+System.identityHashCode(proxy)+" target: "+target.getClass());
      return interfaceClass.cast(proxy);
   }

}
