/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.spi;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.virtual.VirtualFile;

/**
 * A profile represents a named collection of deployments on a server
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision$
 */
public interface Profile
{
   /**
    * Get the key used to create the Profile
    * @return the key used to create the Profile
    */
   ProfileKey getKey();

   /**
    * The x.y.z version of the profile
    * 
    * @return the version if known, null if its unspecified.
    */
   String getVersion();

   /**
    * Get the system time in milliseconds the profile was last modified. 
    * @return System.currentTimeMillis of last modification
    */
   long getLastModified();

   /**
    * Get the names of the deployments in the profile
    * @return names of deployments
    */
   Set<String> getDeploymentNames();

   /**
    * Get the names of the deployments for the given phase defined in this profile
    * @param phase - the phase of the deployment as it relates to when the
    * deployment is loaded
    * @return names of deployments
    */
   Set<String> getDeploymentNames(DeploymentPhase phase);

   //public DeploymentInfo getDeploymentInfo(String name);

   /**
    * Get the names of the deployment in the profile that have the
    * given deployment type.
    * 
    * @param type - the deployment type
    * @return the deployments names by type
    */
   Set<String> getDeploymentNamesForType(String type);

   /**
    * Get the root for deployments of the given phase.
    * 
    * @param phase the deployment phase
    * @return the VirtualFile directory that contains deployments of the
    * given phase.
    * @throws Exception for any error
    */
   VirtualFile getRootFile(DeploymentPhase phase) throws Exception;

   /**
    * Add a deployment
    * 
    * @param d the deployment
    * @param phase - the phase of the deployment as it relates to when the
    * deployment is loaded
    * @throws Exception for any error
    */
   void addDeployment(VFSDeployment d, DeploymentPhase phase) throws Exception;
   
   /**
    * Update a deployment. This can be used to save attachments, other
    * metadata of the deployment.
    * 
    * @param d the deployment
    * @param phase - the phase of the deployment as it relates to when the
    * deployment is loaded
    * @throws Exception for any error
    */
   void updateDeployment(VFSDeployment d, DeploymentPhase phase) throws Exception;

   /**
    * Remove a deployment
    * 
    * @param name the deployment name
    * @param phase - the phase of the deployment as it relates to when the
    * deployment is loaded
    * @return the DeploymentContext for the name if found
    * @throws Exception for any error
    */
   VFSDeployment removeDeployment(String name, DeploymentPhase phase) throws Exception;

   /**
    * Get a named deployment.
    * 
    * @param name - the deployment name
    * @param phase - the phase of the deployment as it relates to when the
    * deployment is loaded. If null, then all phases are queried.
    * @return the named bootstrap
    * @throws Exception for any error
    * @throws NoSuchDeploymentException - if there is no such bootstrap
    */
   VFSDeployment getDeployment(String name, DeploymentPhase phase) throws Exception, NoSuchDeploymentException;

   /**
    * @param phase - the phase of the deployment as it relates to when the
    * deployment is loaded
    * @return the bootstrap instances in this profile.
    * @throws Exception for any error
    */
   Collection<VFSDeployment> getDeployments(DeploymentPhase phase) throws Exception;

   /**
    * Get all deployments defined in this profile
    * 
    * @return the deployment instances in this profile.
    * @throws Exception for any error
    */
   Collection<VFSDeployment> getDeployments() throws Exception;

   /**
    * Get the deployments that have been modified since the last
    * check. 
    * 
    * @return the modified deployments, empty if none exist or
    * the profile does not support hot deployments.
    * 
    * @throws Exception - thrown on error scanning for deployments
    */
   Collection<ModificationInfo> getModifiedDeployments() throws Exception;

   /**
    * Enable/disable the getModifiedDeployments results. This can be
    * used to disable the getModifiedDeployments results during
    * periods such as bootstrap where changes should be delayed.
    * @see #getModifiedDeployments
    * @param flag - the enable/disable flag
    */
   void enableModifiedDeploymentChecks(boolean flag);

   /**
    * Get the config
    * 
    * @return the config
    */
   Map<String, Object> getConfig();

   /**
    * Checks whether a deployment context is available in the basic profile service.
    * 
    * @param    name    the name of the deployment
    * @param    phase   optionally a deployment phase, if null search all
    * @return   true if the deployment is found or false otherwise
    */
   boolean hasDeployment(String name, DeploymentPhase phase);
}
