/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.repository;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipInputStream;

import org.jboss.deployers.spi.attachments.Attachments;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.AttachmentsSerializer;
import org.jboss.profileservice.spi.DeploymentRepository;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.NoSuchDeploymentException;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ModificationInfo.ModifyStatus;
import org.jboss.util.file.Files;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;

/**
 * An implementation of DeploymentRepository that relies on java
 * serialization to store contents on the file system.
 * 
 * + root/{name}/bootstrap - the bootstrap beans
 * + root/{name}/deployers - profile deployers
 * + root/{name}/deploy - installed deployments
 * + root/{name}/apps - post install deployments
 * + root/{name}/attachments - pre-processed attachments + admin edits to deployments
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 69702 $
 */
public class SerializableDeploymentRepository
   implements DeploymentRepository
{
   private static final Logger log = Logger.getLogger(SerializableDeploymentRepository.class);

   /** The server root container the deployments */
   private File root;
   /** The bootstrap jboss-service.xml dir */
   private File bootstrapDir;
   /** The server static libraries */
   private File libDir;
   /** The deployers phase deployments dir */
   private File deployersDir;
   /** The application phase deployments dir */
   private File[] applicationDirs;
   /** The deployment post edit */
   private File adminEditsRoot;
   /** The profile key this repository is associated with */
   private ProfileKey key;
   /** The bootstrap VFSDeployments */
   private LinkedHashMap<String,VFSDeployment> bootstrapCtxs = new LinkedHashMap<String,VFSDeployment>();
   /** The deployer VFSDeployments */
   private LinkedHashMap<String,VFSDeployment> deployerCtxs = new LinkedHashMap<String,VFSDeployment>();
   /** The application VFSDeployments */
   private LinkedHashMap<String,VFSDeployment> applicationCtxs = new LinkedHashMap<String,VFSDeployment>();
   /** The {@link VFSDeployment#getTransientManagedObjects()} serializer */
   private AttachmentsSerializer serializer;
   /** The last time the profile was modified */
   private long lastModified;

   public SerializableDeploymentRepository(File root, URI[] appURIs, ProfileKey key)
   {
      this.root = root;
      this.key = key;
      this.setApplicationURIs(appURIs);
   }

   public AttachmentsSerializer getSerializer()
   {
      return serializer;
   }
   public void setSerializer(AttachmentsSerializer serializer)
   {
      this.serializer = serializer;
   }

   public URI[] getApplicationURIs()
   {
      URI[] appURIs = new URI[applicationDirs.length];
      for (int n = 0; n < applicationDirs.length; n ++)
      {
         File applicationDir = applicationDirs[n];
         appURIs[n] = applicationDir.toURI();
      }
      return appURIs;
   }
   public void setApplicationURIs(URI[] uris)
   {
      applicationDirs = new File[uris.length];
      for (int n = 0; n < uris.length; n ++)
      {
         URI uri = uris[n];
         applicationDirs[n] = new File(uri);
      }
   }

   public boolean exists()
   {
      File profileRoot = new File(root, key.getName());
      return profileRoot.exists();
   }

   public long getLastModified()
   {
      return this.lastModified;
   }

   public URI getDeploymentURI(DeploymentPhase phase)
   {
      URI uri = null;
      switch( phase )
      {
         case BOOTSTRAP:
            uri = this.getBootstrapURI();
            break;
         case DEPLOYER:
            uri = this.getDeployersURI();
            break;
         case APPLICATION:
            uri = this.getApplicationURI();
            break;
      }
      return uri;
   }
   public void setDeploymentURI(URI uri, DeploymentPhase phase)
   {
      switch( phase )
      {
         case BOOTSTRAP:
            this.setBootstrapURI(uri);
            break;
         case DEPLOYER:
            this.setDeployersURI(uri);
            break;
         case APPLICATION:
            this.setApplicationURIs(new URI[]{uri});
            break;
      }
   }
   public Set<String> getDeploymentNames()
   {
      HashSet<String> names = new HashSet<String>();
      names.addAll(bootstrapCtxs.keySet());
      names.addAll(deployerCtxs.keySet());
      names.addAll(applicationCtxs.keySet());
      return names;
   }
   public Set<String> getDeploymentNames(DeploymentPhase phase)
   {
      HashSet<String> names = new HashSet<String>();
      switch( phase )
      {
         case BOOTSTRAP:
            names.addAll(this.bootstrapCtxs.keySet());
            break;
         case DEPLOYER:
            names.addAll(this.deployerCtxs.keySet());
            break;
         case APPLICATION:
            names.addAll(this.applicationCtxs.keySet());
            break;
      }
      return names;      
   }

   public Set<String> getDeploymentNamesForType(String type)
   {
      HashSet<String> names = new HashSet<String>();
      for(VFSDeployment ctx : bootstrapCtxs.values())
      {
         Set<String> types = ctx.getTypes();
         if( types != null && types.contains(type) )
            names.add(ctx.getName());
      }
      for(VFSDeployment ctx : deployerCtxs.values())
      {
         Set<String> types = ctx.getTypes();
         if( types != null && types.contains(type) )
            names.add(ctx.getName());
      }
      for(VFSDeployment ctx : applicationCtxs.values())
      {
         Set<String> types = ctx.getTypes();
         if( types != null && types.contains(type) )
            names.add(ctx.getName());
      }
      return names;
   }

   public void addDeploymentContent(String name, ZipInputStream contentIS, DeploymentPhase phase)
      throws IOException
   {
   }
   public void addDeployment(String vfsPath, VFSDeployment d, DeploymentPhase phase)
      throws Exception
   {
      switch( phase )
      {
         case BOOTSTRAP:
            this.addBootstrap(vfsPath, d);
            break;
         case DEPLOYER:
            this.addDeployer(vfsPath, d);
            break;
         case APPLICATION:
            this.addApplication(vfsPath, d);
            break;
      }
   }

   public void updateDeployment(String vfsPath, VFSDeployment d, DeploymentPhase phase)
      throws Exception
   {
   }
   public VFSDeployment getDeployment(String name, DeploymentPhase phase)
      throws Exception, NoSuchDeploymentException
   {
      VFSDeployment ctx = null;
      if( phase == null )
      {
         // Try all phases
         try
         {
            ctx = this.getBootstrap(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
         try
         {
            if( ctx == null )
               ctx = this.getDeployer(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
         try
         {
            if( ctx == null )
               ctx = this.getApplication(name);
         }
         catch(NoSuchDeploymentException ignore)
         {
         }
      }
      else
      {
         switch( phase )
         {
            case BOOTSTRAP:
               ctx = this.getBootstrap(name);
               break;
            case DEPLOYER:
               ctx = this.getDeployer(name);
               break;
            case APPLICATION:
               ctx = this.getApplication(name);
               break;
         }
      }
      // Make sure we don't return null
      if( ctx == null )
         throw new NoSuchDeploymentException("name="+name+", phase="+phase);
      return ctx;
   }

   public Collection<VFSDeployment> getDeployments()
   {
      HashSet<VFSDeployment> deployments = new HashSet<VFSDeployment>();
      deployments.addAll(this.bootstrapCtxs.values());
      deployments.addAll(this.deployerCtxs.values());
      deployments.addAll(this.applicationCtxs.values());
      return Collections.unmodifiableCollection(deployments);
   }

   /**
    * Scan the applications for changes.
    */
   public synchronized Collection<ModificationInfo> getModifiedDeployments()
      throws Exception
   {
      ArrayList<ModificationInfo> modified = new ArrayList<ModificationInfo>();
      Collection<VFSDeployment> apps = getApplications();
      boolean trace = log.isTraceEnabled();
      if( trace )
         log.trace("Checking applications for modifications");
      if( apps != null )
      {
         Iterator<VFSDeployment> iter = apps.iterator();
         while( iter.hasNext() )
         {
            VFSDeployment ctx = iter.next();
            VirtualFile root = ctx.getRoot();
            Long rootLastModified = root.getLastModified();
            String name = root.getPathName();
            // Check for removal
            if( root.exists() == false )
            {
               ModificationInfo info = new ModificationInfo(ctx, rootLastModified, ModifyStatus.REMOVED);
               modified.add(info);
               iter.remove();
               if( trace )
                  log.trace(name+" was removed");
            }
            // Check for modification
            else if( root.hasBeenModified() )
            {
               if( trace )
                  log.trace(name+" was modified: "+rootLastModified);
               // Need to create a duplicate ctx
               VFSDeployment ctx2 = loadDeploymentData(root);               
               ModificationInfo info = new ModificationInfo(ctx2, rootLastModified, ModifyStatus.MODIFIED);
               modified.add(info);
            }
            // TODO: this could check metadata files modifications as well
         }
         // Now check for additions
         for (File applicationDir : applicationDirs)
         {
            VFS deployVFS = VFS.getVFS(applicationDir.toURI());
            VirtualFile deployDir = deployVFS.getRoot();
            List<VirtualFile> children = deployDir.getChildren();
            for(VirtualFile vf : children)
            {
               URI uri = vf.toURI();
               if( applicationCtxs.containsKey(uri.toString()) == false )
               {
                  VFSDeployment ctx = loadDeploymentData(vf);
                  ModificationInfo info = new ModificationInfo(ctx, vf.getLastModified(), ModifyStatus.ADDED);
                  modified.add(info);
                  applicationCtxs.put(vf.getName(), ctx);
               }
            }
         }
      }
      if(modified.size() > 0)
         lastModified = System.currentTimeMillis();
      return modified;
   }

   public Collection<VFSDeployment> getDeployments(DeploymentPhase phase)
      throws Exception
   {
      Collection<VFSDeployment> ctxs = null;
      switch( phase )
      {
         case BOOTSTRAP:
            ctxs = this.getBootstraps();
            break;
         case DEPLOYER:
            ctxs = this.getDeployers();
            break;
         case APPLICATION:
            ctxs = this.getApplications();
            break;
      }
      return ctxs;
   }

   public VFSDeployment removeDeployment(String name, DeploymentPhase phase)
      throws Exception
   {
      VFSDeployment ctx = null;
      switch( phase )
      {
         case BOOTSTRAP:
            ctx = this.removeBootstrap(name);
            break;
         case DEPLOYER:
            ctx = this.removeDeployer(name);
            break;
         case APPLICATION:
            ctx = this.removeApplication(name);
            break;
      }
      return ctx;
   }
   public String toString()
   {
      StringBuilder tmp = new StringBuilder(super.toString());
      tmp.append("(root=");
      tmp.append(root);
      tmp.append(", key=");
      tmp.append(key);
      tmp.append(")");
      return tmp.toString();
   }

   /**
    * Create a profile deployment repository
    * 
    * @throws IOException
    */
   public void create() throws Exception
   {
      File profileRoot = new File(root, key.getName());
      if( profileRoot.exists() == true )
         throw new IOException("Profile root already exists: "+profileRoot);
      if( profileRoot.mkdirs() == false )
         throw new IOException("Failed to create profile root: "+profileRoot);
      // server/{name}/bootstrap
      bootstrapDir = new File(profileRoot, "bootstrap");
      if( bootstrapDir.mkdirs() == false )
         throw new IOException("Failed to create profile bootstrap dir: "+bootstrapDir);

      // server/{name}/deployers
      deployersDir = new File(profileRoot, "deployers");
      if( deployersDir.mkdirs() == false )
         throw new IOException("Failed to create profile deployers dir: "+deployersDir);

      // server/{name}/deploy
      for (File applicationDir : applicationDirs)
      {
         if( applicationDir.mkdirs() == false )
            throw new IOException("Failed to create profile deploy dir: "+applicationDir);
      }
      // server/{name}/lib
      libDir = new File(profileRoot, "lib");
      if( libDir.mkdirs() == false )
         throw new IOException("Failed to create profile lib dir: "+libDir);

      adminEditsRoot = new File(profileRoot, "profile/edits");
      if( adminEditsRoot.mkdirs() == false )
         throw new IOException("Failed to create profile adminEdits dir: "+adminEditsRoot);
   }

   /**
    * Load the profile deployments
    * 
    * @throws IOException
    * @throws NoSuchProfileException
    */
   public void load() throws Exception, NoSuchProfileException
   {
      if( serializer == null )
         throw new IllegalStateException("serializer has not been set");

      File profileRoot = new File(root, key.getName());
      if( profileRoot.exists() == false )
         throw new NoSuchProfileException("Profile root does not exists: "+profileRoot);
      // server/{name}/bootstrap
      bootstrapDir = new File(profileRoot, "bootstrap");
      if( bootstrapDir.exists() == false )
      {
         //throw new FileNotFoundException("Profile contains no bootstrap dir: "+bootstrapDir);
         // fallback to conf/jboss-service.xml for now
         bootstrapDir = null;
      }

      // server/{name}/deployers
      deployersDir = new File(profileRoot, "deployers");
      if( deployersDir.exists() == false )
         throw new FileNotFoundException("Profile contains no deployers dir: "+deployersDir);

      // server/{name}/deploy
      for (File applicationDir : applicationDirs)
      {
         if( applicationDir.exists() == false )
            throw new FileNotFoundException("Profile contains no deploy dir: "+applicationDir);
      }

      adminEditsRoot = new File(profileRoot, "profile/edits");

      if( bootstrapDir != null )
      {
         VFS bootstrapVFS = VFS.getVFS(bootstrapDir.toURI());
         loadBootstraps(bootstrapVFS.getRoot());
      }
      else
      {
         // hack to load conf/jboss-service.xml until its removed
         loadBootstraps(null);         
      }
      VFS deployersVFS = VFS.getVFS(deployersDir.toURI());
      loadDeployers(deployersVFS.getRoot());
      for (File applicationDir : applicationDirs)
      {
         VFS deployVFS = VFS.getVFS(applicationDir.toURI());
         loadApplications(deployVFS.getRoot());
      }
      this.lastModified = System.currentTimeMillis();
   }

   /**
    * Remove the contents of the profile repository
    * @throws IOException
    * @throws NoSuchProfileException
    */
   public void remove() throws IOException, NoSuchProfileException
   {
      File profileRoot = new File(root, key.getName());
      Files.delete(profileRoot);
   }

   protected void addBootstrap(String vfsPath, VFSDeployment ctx)
      throws Exception
   {
      this.bootstrapCtxs.put(vfsPath, ctx);
   }

   // Managed object attachments for a deployment
   public void addManagedObject(String vfsPath, Attachments edits)
      throws IOException
   {
      Map<String, Object> map = edits.getAttachments();
      File attachments = new File(adminEditsRoot, vfsPath+".edits");
      FileOutputStream fos = new FileOutputStream(attachments);
      ObjectOutputStream oos = new ObjectOutputStream(fos);
      oos.writeObject(map);
      oos.close();
      fos.close();
      lastModified = System.currentTimeMillis();
   }

   protected void addDeployer(String vfsPath, VFSDeployment ctx)
      throws Exception
   {
      this.deployerCtxs.put(vfsPath, ctx);
   }

   protected void addApplication(String vfsPath, VFSDeployment ctx)
      throws Exception
   {
      this.applicationCtxs.put(vfsPath, ctx);
   }

   protected VFSDeployment getBootstrap(String vfsPath)
      throws Exception
   {
      VFSDeployment ctx = bootstrapCtxs.get(vfsPath);
      if( ctx == null )
         throw new NoSuchDeploymentException(vfsPath);
      return ctx;
   }

   protected Collection<VFSDeployment> getBootstraps()
      throws Exception
   {
      Collection<VFSDeployment> ctxs = bootstrapCtxs.values();
      return ctxs;
   }

   protected URI getBootstrapURI()
   {
      return bootstrapDir.toURI();
   }
   protected URI getDeployersURI()
   {
      return deployersDir.toURI();
   }
   protected URI getApplicationURI()
   {
      File applicationDir = applicationDirs[0];
      return applicationDir.toURI();
   }
   protected VFSDeployment getDeployer(String vfsPath)
      throws Exception
   {
      VFSDeployment ctx = deployerCtxs.get(vfsPath);
      if( ctx == null )
         throw new NoSuchDeploymentException(vfsPath);
      return ctx;
   }

   protected Collection<VFSDeployment> getDeployers()
      throws Exception
   {
      Collection<VFSDeployment> ctxs = deployerCtxs.values();
      return ctxs;
   }

   protected VFSDeployment getApplication(String vfsPath)
      throws Exception
   {
      VFSDeployment ctx = applicationCtxs.get(vfsPath);
      if( ctx == null )
         throw new NoSuchDeploymentException(vfsPath);
      return ctx;
   }

   protected Collection<VFSDeployment> getApplications()
      throws Exception
   {
      Collection<VFSDeployment> ctxs = applicationCtxs.values();
      return ctxs;
   }

   protected VFSDeployment removeBootstrap(String vfsPath) throws IOException
   {
      VFSDeployment vfsDeployment = bootstrapCtxs.get(vfsPath);
      if(vfsDeployment == null)
         throw new IllegalStateException("Deployment not found: " + vfsPath);
      File bootstrapFile = new File(bootstrapDir, vfsDeployment.getSimpleName());
      if( bootstrapFile.delete() == false )
         throw new IOException("Failed to delete: "+bootstrapFile);
      return bootstrapCtxs.remove(vfsPath);
   }

   // this is an infinite loop
   protected VFSDeployment removeDeployer(String vfsPath) throws IOException
   {
      VFSDeployment vfsDeployment = deployerCtxs.get(vfsPath);
      if(vfsDeployment == null)
         throw new IllegalStateException("Deployment not found: " + vfsPath);
      File deployerFile = new File(deployersDir, vfsDeployment.getSimpleName());
      if( Files.delete(deployerFile) == false )
         throw new IOException("Failed to delete: "+deployerFile);
      return deployerCtxs.remove(vfsPath);
   }
   protected VFSDeployment removeApplication(String vfsPath) throws IOException
   {
      VFSDeployment vfsDeployment = applicationCtxs.get(vfsPath);
      if(vfsDeployment == null)
         throw new IllegalStateException("Deployment not found: " + vfsPath);
      // Find the application dir
      File applicationDir = applicationDirs[0];
      File deploymentFile = new File(applicationDir, vfsDeployment.getSimpleName());
      if( Files.delete(deploymentFile) == false )
         throw new IOException("Failed to delete: "+deploymentFile);
      return this.applicationCtxs.remove(vfsPath);
   }
   protected void setBootstrapURI(URI uri)
   {
      bootstrapDir = new File(uri);
   }
   protected void setDeployersURI(URI uri)
   {
      deployersDir = new File(uri);
   }

   /**
    * Load the bootstrap descriptors under bootstrapDir:
    * 
    * @param bootstrapDir
    * @throws IOException
    */
   private void loadBootstraps(VirtualFile bootstrapDir)
      throws IOException
   {
      if( bootstrapDir != null )
      {
         List<VirtualFile> children = bootstrapDir.getChildren();
         for(VirtualFile vf : children)
         {
            VFSDeployment vfCtx = loadDeploymentData(vf);
            bootstrapCtxs.put(vf.getName(), vfCtx);       
         }
      }
      else
      {
         // fallback to conf/jboss-service.xml for now
         File profileRoot = new File(root, key.getName());
         File confDir = new File(profileRoot, "conf");
         VirtualFile confVF = VFS.getRoot(confDir.toURI());
         VirtualFile jbossServiceVF = confVF.findChild("jboss-service.xml");
         VFSDeployment vfCtx = loadDeploymentData(jbossServiceVF);
         bootstrapCtxs.put(vfCtx.getName(), vfCtx);                
      }
   }

   /**
    * Load all the deployments under the deployersDir.
    * 
    * @param deployersDir
    * @throws IOException
    */
   private void loadDeployers(VirtualFile deployersDir)
      throws IOException
   {
      List<VirtualFile> children = deployersDir.getChildren();
      for(VirtualFile vf : children)
      {
         VFSDeployment vfCtx = loadDeploymentData(vf);
         deployerCtxs.put(vfCtx.getName(), vfCtx);       
      }
   }

   /**
    * Load all the applications under the applicationDir.
    * 
    * @param applicationDir
    * @throws IOException
    */
   private void loadApplications(VirtualFile applicationDir)
      throws IOException
   {
      List<VirtualFile> children = applicationDir.getChildren();
      for(VirtualFile vf : children)
      {
         VFSDeployment vfCtx = loadDeploymentData(vf);
         applicationCtxs.put(vfCtx.getName(), vfCtx);         
      }
   }

   /**
    * TODO: this could be dropped since the serialize aspect loads the data
    * @param file
    * @return the deployment
    */
   private VFSDeployment loadDeploymentData(VirtualFile file)
   {
      // Check for a persisted context
      // Load the base deployment
      VFSDeployment deployment = VFSDeploymentFactory.getInstance().createVFSDeployment(file);
      log.debug("Created deployment: "+deployment);
      return deployment;
   }

}
