/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.repository;

import java.io.IOException;
import java.util.Collection;

import org.jboss.deployers.spi.management.ManagementView;
import org.jboss.deployers.spi.management.deploy.DeploymentManager;
import org.jboss.logging.Logger;
import org.jboss.profileservice.spi.DeploymentRepository;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileRepository;
import org.jboss.profileservice.spi.ProfileService;
import org.jboss.system.server.profile.repository.ProfileImpl;


/**
 * A ProfileService impl that uses a ProfileRepository/DeploymentRepository
 * based Profile implementation.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision$
 */
public class ProfileServiceImpl
   implements ProfileService
{
   private static final Logger log = Logger.getLogger(ProfileServiceImpl.class);
   /** The name of the profile to use */
   private String name;
   /** The profile repository plugin */
   private ProfileRepository profileRepository;
   /** The deployment repository plugin for the current profile */
   private DeploymentRepository deploymentRepository;
   /** The active profile */
   private Profile profile;
   /** The management view for the active profile */
   private ManagementView mgtView;
   /** The profiles DeploymentManager */
   private DeploymentManager deployMgr;
   /** The version of the profile to use */
   private String version;

   public ProfileServiceImpl(String name) throws IOException
   {
      this.name = name;
   }
   // Properties ------------------------

   public String getName()
   {
      return this.name;
   }
   public void setName(String name)
   {
      this.name = name;
   }

   public String getVersion()
   {
      return version;
   }

   public void setVersion(String version)
   {
      this.version = version;
   }

   public ProfileRepository getProfileRepository()
   {
      return profileRepository;
   }

   public void setProfileRepository(ProfileRepository profileRepository)
   {
      this.profileRepository = profileRepository;
   }

   public DeploymentRepository getDeploymentRepository()
   {
      return deploymentRepository;
   }

   public void setDeploymentRepository(DeploymentRepository repository)
   {
      this.deploymentRepository = repository;
   }

   public void start()
      throws Exception
   {
      if( profileRepository == null )
         throw new IllegalStateException("No profileRepository specified");
      // Obtain the deployment repository for the profile
      ProfileKey key = new ProfileKey(name);
      Profile profile = this.getProfile(key);
      log.info("Loaded profile: "+profile);
   }

   // ProfileService implementation --------------------
   /**
    * There is only the default domain (ProfileKey.DEFAULT)
    */
   public String[] getDomains()
   {
      String[] domains = {ProfileKey.DEFAULT};
      return domains;
   }

   /**
    * Get the keys for all known profiles
    * @return keys for all known profiles
    */
   public Collection<ProfileKey> getProfileKeys()
   {
      Collection<ProfileKey> keys = profileRepository.getProfileKeys();
      return keys;
   }

   /**
    * Always returns the default profile.
    */
   public Profile getProfile(ProfileKey key)
      throws NoSuchProfileException
   {
      if( profileRepository == null )
         throw new IllegalStateException("No profileRepository specified");
      if(profile != null)
      {
         if(key.isDefaultKey() || profile.getKey().equals(key))
            return profile;
      }

      // Obtain the deployment repository for the profile
      try
      {
         deploymentRepository = profileRepository.getProfileDeploymentRepository(key);
      }
      catch(NoSuchProfileException e)
      {
         throw e;
      }
      catch(Exception e)
      {
         throw new NoSuchProfileException("Failed to obtain deployment repository for: "+key, e);
      }
      log.info("Loading profile: "+name+" from: "+deploymentRepository);
      profile = new ProfileImpl(deploymentRepository, key);
      return profile;
   }
   public Profile getActiveProfile()
      throws NoSuchProfileException
   {
      return profile;   
   }

   public String[] getProfileDeploymentNames(ProfileKey key)
      throws NoSuchProfileException
   {
      String[] names = {"default"};
      return names;
   }

   public ManagementView getViewManager()
   {
      return mgtView;
   }
   public void setViewManager(ManagementView mgtView)
   {
      this.mgtView = mgtView;
   }

   public DeploymentManager getDeploymentManager()
   {
      return deployMgr;
   }
   public void setDeploymentManager(DeploymentManager deployMgr)
   {
      this.deployMgr = deployMgr;
   }

   public Profile newProfile(ProfileKey key)
      throws Exception
   {
      DeploymentRepository repo = profileRepository.createProfileDeploymentRepository(key);
      return new ProfileImpl(repo, key);
   }

   public void removeProfile(ProfileKey key)
      throws Exception, NoSuchProfileException
   {
      profileRepository.removeProfileDeploymentRepository(key);
   }
}

