/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.server.profileservice.repository;

import java.io.File;
import java.util.Collection;
import java.util.HashSet;

import org.jboss.deployers.client.spi.main.MainDeployer;
import org.jboss.profileservice.spi.DeploymentRepository;
import org.jboss.profileservice.spi.DeploymentRepositoryFactory;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileRepository;

/**
 * An implementation of ProfileRepository that uses the following
 * file system structure for storing deployments and admin
 * edits.
 * 
 * root-file
 * + server/{name}/bootstrap - bootstrap beans/services
 * + server/{name}/deployers - profile deployers
 * + server/{name}/deploy - installed deployments
 * + server/{name}/apps - post install deployments
 * + server/{name}/admin - admin edits to deployments
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 81070 $
 */
public class FileProfileRepository
   implements ProfileRepository
{
   private File root;
   DeploymentRepositoryFactory repositoryFactory;
   private MainDeployer mainDeployer;

   public FileProfileRepository()
   {
      this(null);
   }
   public FileProfileRepository(File root)
   {
      this.root = root;
   }

   public DeploymentRepositoryFactory getRepositoryFactory()
   {
      return repositoryFactory;
   }
   public void setRepositoryFactory(DeploymentRepositoryFactory repositoryFactory)
   {
      this.repositoryFactory = repositoryFactory;
   }
   public MainDeployer getMainDeployer()
   {
      return mainDeployer;
   }
   public void setMainDeployer(MainDeployer mainDeployer)
   {
      this.mainDeployer = mainDeployer;
   }

   /**
    * Get the server profile store root directory.
    * @return the server profile root directory containing the
    * available profiles.
    */
   public File getStoreRoot()
   {
      return root;
   }
   /**
    * Set the profiles store root.
    * @param root - the server profile root directory containing the
    * available profiles.
    */
   public void setStoreRoot(File root)
   {
      this.root = root;
   }

   /**
    * Get the keys for all known profiles. This lists the root store to
    * obtain the server profiles.
    * @return keys for all known profiles
    */
   public Collection<ProfileKey> getProfileKeys()
   {
      Collection<ProfileKey> keys = new HashSet<ProfileKey>();
      for(File profile : root.listFiles())
      {
         keys.add(new ProfileKey(profile.getName()));
      }
      return keys;
   }

   /**
    * 
    */
   public DeploymentRepository createProfileDeploymentRepository(ProfileKey key)
      throws Exception
   {
      DeploymentRepository repository = repositoryFactory.getDeploymentRepository(key);
      repository.create();
      return repository;
   }

   /**
    * Load an existing profile deployment repository.
    */
   public DeploymentRepository getProfileDeploymentRepository(ProfileKey key)
      throws Exception, NoSuchProfileException
   {
      DeploymentRepository repository = repositoryFactory.getDeploymentRepository(key);
      repository.load();
      return repository;
   }

   public void removeProfileDeploymentRepository(ProfileKey key)
      throws Exception, NoSuchProfileException
   {      
      DeploymentRepository repository = repositoryFactory.getDeploymentRepository(key);
      repository.remove();  
   }

}
