/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.system.server.profileservice.repository;

import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.virtual.VirtualFile;

/**
 * The profile deployment.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class AbstractProfileDeployment implements ProfileDeployment
{
   
   /** The name. */
   private String name;
   
   /** The vfs root. */
   private VirtualFile root;

   /** The attachments. */
   private Map<String, Object> attachments = new ConcurrentHashMap<String, Object>();
   
   /** The serialVersionUID. */
   private static final long serialVersionUID = -2600392045205267112L;
   
   /**
    * Get the vfs file name safely.
    * @see VFSDeployment
    * 
    * @param root the virtual file
    * @return the name
    */
   static final String safeVirtualFileName(VirtualFile root)
   {
      if (root == null)
         throw new IllegalArgumentException("Null root");

      try
      {
         return root.toURI().toString();
      }
      catch (Exception e)
      {
         return root.getName();
      }
   }
   
   public AbstractProfileDeployment(String name)
   {
      if(name == null)
         throw new IllegalArgumentException("Null name.");
      this.name = name;
   }
   
   public AbstractProfileDeployment(VirtualFile root)
   {
      this(safeVirtualFileName(root));
      this.root = root;
   }

   /**
    * Get the deployment name.
    * 
    * @return the name
    */
   public String getName()
   {
      return this.name;
   }

   /**
    * Get the root of the deployment.
    * 
    * @return the root, or null if it's not a VFS deployment.
    */
   public VirtualFile getRoot()
   {
      return this.root;
   }
   
   /**
    * Get attachment.
    * 
    * @param name the name of the attachment
    * @return the attachment or null if not present
    * 
    * @throws IllegalArgumentException for a null name
    */
   public Object getAttachment(String name)
   {
      if(name == null)
         throw new IllegalArgumentException("Null attachment name.");
      
      return this.attachments.get(name);
   }

   /**
    * Add attachment.
    *
    * @param name the name of the attachment
    * @param attachment the attachment
    * @return any previous attachment
    * 
    * @throws IllegalArgumentException for a null name or attachment
    */
   public Object addAttachment(String name, Object attachment)
   {
      if(name == null)
         throw new IllegalArgumentException("Null attachment name.");
      if(attachment == null)
         throw new IllegalArgumentException("Null attachment.");
      
      return this.attachments.put(name, attachment);
   }
   
   /**
    * Get all attachments.
    * 
    * @return the attachments
    */
   public Map<String, Object> getAttachments()
   {
      return Collections.unmodifiableMap(this.attachments);
   }
   
   public String toString()
   {
      return "AbstractProfileDeployment(" + root != null ? root.getName() : name + ")";
   }
   
}
