/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.system.server.profileservice.repository;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.SyncFailedException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.zip.ZipInputStream;

import org.jboss.deployers.vfs.spi.structure.modified.StructureModificationChecker;
import org.jboss.profileservice.spi.DeploymentContentFlags;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ModificationInfo.ModifyStatus;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;

/**
 * A mutable deployment repository, with hot deployment capabilities.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 * @version $Revision: 85526 $
 */
public class MutableDeploymentRepository extends AbstractDeploymentRepository
{
   /** Should an attempt to overwrite existing content fail in {@link #addDeploymentContent(String, ZipInputStream)}*/
   private boolean failIfAlreadyExists = false;

   /** The structure modification checker */
   private StructureModificationChecker checker;

   /** A lock for the hot deployment/{@link #getModifiedDeployments()} */
   private ReentrantReadWriteLock contentLock = new ReentrantReadWriteLock(true);
   
   public MutableDeploymentRepository(ProfileKey key, URI[] uris)
   {
      super(key, uris);
   }

   /**
    * Get the structure modified checker.
    *
    * @return the checker
    */
   protected StructureModificationChecker getChecker()
   {
      if (checker == null)
         throw new IllegalArgumentException("Checker must be set");

      return checker;
   }

   /**
    * Set the checker.
    *
    * @param checker the checker
    */
   public void setChecker(StructureModificationChecker checker)
   {
      this.checker = checker;
   }

   public void load() throws Exception
   {
      for(URI uri : getRepositoryURIs())
      {
         VirtualFile root = getCachedVirtualFile(uri);
         loadApplications(root);
      }
      updateLastModfied();
   }
   
   public String addDeploymentContent(String vfsPath, InputStream contentIS) throws IOException
   {
      boolean trace = log.isTraceEnabled();
      // Suspend hot deployment checking
      if( trace )
         log.trace("Aquiring content write lock");
      contentLock.writeLock().lock();
      String repositoryName = null;
      try
      {
         if(getRepositoryURIs().length == 0)
            throw new IllegalStateException("No uris associated with this repository.");
         
         // FIXME, this is writing the content to the first URI
         // Write the content out
         File contentRoot = new File(getRepositoryURIs()[0]); 
         if(contentRoot == null)
            throw new FileNotFoundException("Failed to obtain content dir for phase: "+vfsPath);
         File contentFile = new File(contentRoot, vfsPath);
         
         if(failIfAlreadyExists && contentFile.exists())
            throw new SyncFailedException("Deployment content already exists: "+contentFile.getAbsolutePath());
         FileOutputStream fos = new FileOutputStream(contentFile);
         try
         {
            byte[] tmp = new byte[4096];
            int read;
            while((read = contentIS.read(tmp)) > 0)
            {
               if (trace)
                  log.trace("write, " + read);
               fos.write(tmp, 0, read);
            }
            fos.flush();
            // Get the vfs uri and add the VFS uri to the cached VFS uris
            VirtualFile contentVF = VFS.getRoot(contentFile.toURI());
            try
            {
               contentVF = getCachedVirtualFile(contentVF.toURI());
               repositoryName = contentVF.toURI().toString();
            }
            catch(URISyntaxException e)
            {
               throw new RuntimeException(e); // This should not happen anyway
            }
         }
         finally
         {
            try
            {
               fos.close();
            }
            catch (IOException ignored)
            {
            }
         }

         // Lock the content
         lockDeploymentContent(repositoryName);
      }
      finally
      {
         // Allow hot deployment checking
         contentLock.writeLock().unlock();
         if(trace)
            log.trace("Released content write lock");
      }
      return repositoryName;
   }

   public synchronized Collection<ModificationInfo> getModifiedDeployments() throws Exception
   {
      ArrayList<ModificationInfo> modified = new ArrayList<ModificationInfo>();
      Collection<ProfileDeployment> apps = getDeployments();
      boolean trace = log.isTraceEnabled();
      if (trace)
         log.trace("Checking applications for modifications");
      if (trace)
         log.trace("Aquiring content read lock");
      contentLock.readLock().lock();
      try
      {
         if (apps != null)
         {
            Iterator<ProfileDeployment> iter = apps.iterator();
            while (iter.hasNext())
            {
               ProfileDeployment ctx = iter.next();
               VirtualFile root = ctx.getRoot();
               String pathName = ctx.getName();
               // Ignore locked or disabled applications
               if (this.hasDeploymentContentFlags(pathName, ignoreFlags))
               {
                  if (trace)
                     log.trace("Ignoring locked application: " + root);
                  continue;
               }
               // Check for removal
               if (root.exists() == false)
               {
                  long rootLastModified = root.getLastModified();
                  ModificationInfo info = new ModificationInfo(ctx, rootLastModified, ModifyStatus.REMOVED);
                  modified.add(info);
                  iter.remove();
                  // Remove last modified cache
                  getChecker().removeStructureRoot(root);
                  if (trace)
                     log.trace(pathName + " was removed");
               }
               // Check for modification
               else if (getChecker().hasStructureBeenModified(root) || hasDeploymentContentFlags(pathName, DeploymentContentFlags.MODIFIED))
               {
                  long rootLastModified = root.getLastModified();
                  if (trace)
                     log.trace(pathName + " was modified: " + rootLastModified);
                  // Create the modification info
                  ModificationInfo info = new ModificationInfo(ctx, rootLastModified, ModifyStatus.MODIFIED);
                  modified.add(info);
               }
            }
            // Now check for additions
            for (URI applicationDir : getRepositoryURIs())
            {
               VirtualFile deployDir = getCachedVirtualFile(applicationDir);
               ArrayList<VirtualFile> added = new ArrayList<VirtualFile>();
               addedDeployments(added, deployDir);
               for (VirtualFile vf : added)
               {
                  // Create deployment
                  ProfileDeployment ctx = createDeployment(vf);
                  // Create modification info
                  ModificationInfo info = new ModificationInfo(ctx, vf.getLastModified(), ModifyStatus.ADDED);
                  // Add
                  modified.add(info);
                  addDeployment(ctx.getName(), ctx);
                  getChecker().addStructureRoot(vf);
               }
            }
         }
      }
      finally
      {
         contentLock.readLock().unlock();
         if (trace)
            log.trace("Released content read lock");
      }

      if (modified.size() > 0)
         updateLastModfied();
      return modified;
   }

   public ProfileDeployment removeDeployment(String vfsPath) throws Exception
   {
      // Suspend hot deployment checking
      if( log.isTraceEnabled() )
         log.trace("Aquiring content write lock");
      contentLock.writeLock().lock();
      try
      {
         // Remove the deployment from the filesystem
         ProfileDeployment deployment = getDeployment(vfsPath);
         VirtualFile root = deployment.getRoot();
         if(root.delete() == false)
            throw new IOException("Failed to delete: " + root);
         // Cleanup
         return super.removeDeployment(deployment.getName());
      }
      finally
      {
         contentLock.writeLock().unlock();
         if (log.isTraceEnabled())
            log.trace("Released content write lock");
      }
   }
   
   public void remove() throws Exception
   {
      // FIXME remove
   }
}
