/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.tools;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.jboss.logging.Logger;
import org.jboss.profileservice.spi.ModificationInfo;
import org.jboss.profileservice.spi.MutableProfile;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileService;
import org.jboss.system.server.profileservice.hotdeploy.Scanner;
import org.jboss.system.server.profileservice.repository.MutableDeploymentRepository;
import org.jboss.virtual.VirtualFile;

/**
 * Profile deployment repository adapter.
 *
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class ProfileServiceDeploymentRepositoryAdapter implements DeploymentRepositoryAdapter
{
   private static final Logger log = Logger.getLogger(ProfileServiceDeploymentRepositoryAdapter.class);
   private static final ProfileKey profileName = new ProfileKey("deployment-scanner-profile");

   private Scanner scanner;
   private ProfileService ps;
   private DeploymentScannerProfile profile;

   public ProfileServiceDeploymentRepositoryAdapter(Scanner scanner, ProfileService ps)
   {
      if (scanner == null)
         throw new IllegalArgumentException("Null scanner");
      if (ps == null)
         throw new IllegalArgumentException("Null profile service");

      this.scanner = scanner;
      this.ps = ps;
   }

   /**
    * Create profile.
    *
    * @throws Exception for any error
    */
   public void create() throws Exception
   {
      this.profile = new DeploymentScannerProfile();
      // Create the profile
      registerProfile();
   }

   /**
    * Stop profile.
    */
   public void destroy()
   {
      stopProfile();
   }

   /**
    * Register profile.
    *
    * @throws Exception for any error
    */
   protected void registerProfile() throws Exception
   {
      if(this.ps == null)
         throw new IllegalStateException("Null profile service.");

      // Register
      this.ps.registerProfile(profile);

      // Activate
      log.debug("Activating deployment scanner profile " + profileName);
      this.ps.activateProfile(profileName);
   }

   public void resume()
   {
      scanner.resume();
   }

   public void suspend()
   {
      scanner.suspend();
   }

   /**
    * Stop deactivates and unregisters the transient deployments profile.
    */
   public void stopProfile()
   {
      try
      {
         // Deactivate
         log.debug("Deactivating deployment scanner profile: " + profileName);
         this.ps.deactivateProfile(profileName);
      }
      catch(Exception e)
      {
         log.debug("Failed to deactivate deployment scanner profile: ", e);
      }
      try
      {
         // Unregister
         log.debug("Unregistering transient profile: " + profileName);
         this.ps.unregisterProfile(profileName);
      }
      catch(Exception e)
      {
         log.debug("Failed to unregister deployment scanner profile: ", e);
      }
   }

   public void addURL(URL url) throws URISyntaxException
   {
      profile.getURIs().add(url.toURI());
   }

   public void removeURL(URL url) throws URISyntaxException
   {
      profile.getURIs().remove(url.toURI());
   }

   public boolean hasURL(URL url) throws URISyntaxException
   {
      URI uri = url.toURI();
      // TODO - this only checks this profile
      return profile.getURIs().contains(uri);
   }

   public URL[] listDeployedURLs()
   {
      List<URL> urls = new ArrayList<URL>();

      Collection<ProfileKey> activeProfiles = ps.getActiveProfileKeys();
      if (activeProfiles != null && activeProfiles.isEmpty() == false)
      {
         for (ProfileKey key : activeProfiles)
         {
            // The profile
            Profile profile;
            try
            {
               profile = ps.getActiveProfile(key);
            }
            catch (NoSuchProfileException ignore)
            {
               continue;
            }
            Collection<ProfileDeployment> deployments = profile.getDeployments();
            if (deployments != null && deployments.isEmpty() == false)
            {
               for (ProfileDeployment pd : deployments)
               {
                  VirtualFile root = pd.getRoot();
                  if (root != null)
                  {
                     try
                     {
                        urls.add(root.toURL());
                     }
                     catch (Exception e)
                     {
                        log.warn("Exception while reading root's URL: " + root);
                     }
                  }
               }
            }
         }
      }

      return urls.toArray(new URL[urls.size()]);
   }

   public static class DeploymentScannerProfile extends MutableDeploymentRepository implements MutableProfile
   {
      private volatile boolean enableHotDeployment;

      public DeploymentScannerProfile()
      {
         super(profileName, new URI[0]);
      }

      /**
       * Expose uris.
       *
       * @return the uris
       */
      Collection<URI> getURIs()
      {
         return uris;
      }

      public void addDeployment(ProfileDeployment deployment) throws Exception
      {
         super.addDeployment(deployment.getName(), deployment);
      }

      public void enableModifiedDeploymentChecks(boolean flag)
      {
         this.enableHotDeployment = flag;
      }

      @Override
      public Collection<ModificationInfo> getModifiedDeployments() throws Exception
      {
         if(this.enableHotDeployment == false)
            return Collections.emptySet();

         return super.getModifiedDeployments();
      }

      public ProfileKey getKey()
      {
         return profileName;
      }

      public Collection<ProfileKey> getSubProfiles()
      {
         return Collections.emptySet();
      }

      public boolean hasDeployment(String name)
      {
         // FIXME
         return false;
      }

      public boolean isMutable()
      {
         return true;
      }
   }
}