/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.system.server.profileservice.persistence;

import java.lang.reflect.Proxy;

import javax.management.ObjectName;

import org.jboss.beans.info.spi.BeanInfo;
import org.jboss.beans.info.spi.PropertyInfo;
import org.jboss.logging.Logger;
import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.factory.ManagedObjectFactory;
import org.jboss.managed.spi.factory.InstanceClassFactory;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.metatype.spi.values.MetaMapper;

/**
 * The AttachmentPropertyPopulator, this writes the values of a 
 * ManagedProperty to a attachment.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision: 86914 $
 */
public class AttachmentPropertyPopulator
{

   /** The managed object factory. */
   private ManagedObjectFactory managedObjectFactory;

   /** The meta value factory. */
   private MetaValueFactory metaValueFactory = MetaValueFactory.getInstance();
   
   /** The logger. */
   private static final Logger log = Logger.getLogger(AttachmentPropertyPopulator.class);

   public AttachmentPropertyPopulator()
   {
      this.managedObjectFactory = ManagedObjectFactory.getInstance();
   }
   
   public AttachmentPropertyPopulator(ManagedObjectFactory managedObjectFactory)
   {
      this.managedObjectFactory = managedObjectFactory;
   }
   
   /**
    * Process a ManagedProperty.
    * 
    * @param propertyElement the persisted xml meta data.
    * @param name the property name.
    * @param property the managed property.
    * @param attachment the managed object attachment.
    */
   public void processManagedProperty(String name, ManagedProperty property, Object attachment) throws Throwable
   {
      boolean trace = log.isTraceEnabled();
      PropertyInfo propertyInfo = property.getField(Fields.PROPERTY_INFO, PropertyInfo.class);     
      // Skip not writable properties
      if (propertyInfo == null || propertyInfo.isWritable() == false)
      {
         if (trace)
            log.debug("Skipping not writable property " + propertyInfo);
         return;
      }

      // Get the meta data information
      MetaType metaType = property.getMetaType();
      MetaValue value = property.getValue();

      // FIXME skip CompositeValueInvocationHandler
      MetaMapper mapper = property.getTransientAttachment(MetaMapper.class);
      if (mapper == null && metaType.isComposite())
      {
         if(metaType.getTypeName().equals(ObjectName.class.getName()) == false)
         {
            // unwrap
            Object unwrapped = metaValueFactory.unwrap(value, propertyInfo.getType());
            if (Proxy.isProxyClass(unwrapped.getClass()))
               return;            
         }
      }
      
      // Set value
      InstanceClassFactory icf = managedObjectFactory.getInstanceClassFactory(attachment.getClass(), null);
      BeanInfo beanInfo = propertyInfo.getBeanInfo();
      icf.setValue(beanInfo, property, attachment, value);
      
   }

}
