/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.services.binding.impl;

import java.beans.PropertyEditor;
import java.beans.PropertyEditorManager;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URL;
import java.util.Map;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.jboss.services.binding.ElementServiceBindingValueSource;
import org.jboss.services.binding.ServiceBinding;
import org.jboss.services.binding.URLServiceBindingValueSource;
import org.jboss.util.StringPropertyReplacer;
import org.w3c.dom.Element;

/**
 * A XSLTServiceBindingValueSourceImpl.
 * 
 * @author Brian Stansberry
 * @version $Revision$
 */
public class XSLTServiceBindingValueSourceImpl implements URLServiceBindingValueSource, ElementServiceBindingValueSource
{

   public String getResourceServiceBindingValue(ServiceBinding binding, String input) throws Exception
   {
      if (input == null)
         throw new IllegalArgumentException("input cannot be null");
      
      XSLTServiceBindingValueSourceConfig config = getConfig(binding);
      
      Reader reader = null;
      Writer writer = null;
      try
      {
         reader = Util.getInputStreamReader(input);

         File targetFile = Util.createTempFile();
         writer = new OutputStreamWriter(new FileOutputStream(targetFile));
         
         doXslTransform(binding, config, reader, writer);

         return targetFile.getAbsolutePath();
      }
      finally
      {
         if (reader != null)
            reader.close();
         if (writer != null)
            writer.close();
      }
   }

   public URL getURLServiceBindingValue(ServiceBinding binding, URL input) throws Exception
   {
      if (input == null)
         throw new IllegalArgumentException("input cannot be null");
      
      XSLTServiceBindingValueSourceConfig config = getConfig(binding);
      
      Reader reader = null;
      Writer writer = null;
      try
      {
         reader = Util.getInputStreamReader(input);

         File targetFile = Util.createTempFile();
         writer = new OutputStreamWriter(new FileOutputStream(targetFile));
         
         doXslTransform(binding, config, reader, writer);

         return targetFile.toURL();
      }
      finally
      {
         if (reader != null)
            reader.close();
         if (writer != null)
            writer.close();
      }
   }

   public Element getElementServiceBindingValue(ServiceBinding binding, Element input) throws Exception
   {
      if (input == null)
         throw new IllegalArgumentException("input cannot be null");
      
      PropertyEditor editor = PropertyEditorManager.findEditor(Element.class);
      if (editor == null)
         throw new IllegalStateException("Cannot find PropertyEditor for type Element");
      
      editor.setValue(input);      
      Reader reader = new StringReader(editor.getAsText());
      Writer writer = new StringWriter();
      
      doXslTransform(binding, getConfig(binding), reader, writer);
      
      editor.setAsText(writer.toString());
      return (Element) editor.getValue();
   }

   public Object getServiceBindingValue(ServiceBinding binding, Object... params) throws Exception
   {
      if (params == null || params.length != 1)
      {
        throw new IllegalArgumentException(getClass().getSimpleName() + ".getServiceBindingValue() requires a single-value 'params'");
      }
      
      if (params[0] instanceof String)
      {   
         return getResourceServiceBindingValue(binding, (String) params[0]);
      }
      else if (params[0] instanceof Element)
      {   
         return getElementServiceBindingValue(binding, (Element) params[0]);
      }
      else if (params[0] instanceof URL)
      {   
         return getURLServiceBindingValue(binding, (URL) params[0]);
      }
      
      throw new IllegalArgumentException(getClass().getSimpleName() + ".getServiceBindingValue() requires a single-value 'params' of type String, Element or URL");
   }
   
   private XSLTServiceBindingValueSourceConfig getConfig(ServiceBinding binding)
   {
      Object config = binding.getServiceBindingValueSourceConfig();
      if (config instanceof XSLTServiceBindingValueSourceConfig)
      {
         return (XSLTServiceBindingValueSourceConfig) config;
      }
      else if (config == null)
      {
         throw new IllegalStateException("No config object bound to " + binding);
      }
      throw new IllegalStateException("Incompatible config object of type " + 
                                       config.getClass() + " bound to " + binding +
                                       " -- must use " + XSLTServiceBindingValueSourceConfig.class.getName());
   }

   private void doXslTransform(ServiceBinding binding, XSLTServiceBindingValueSourceConfig config, 
         Reader reader, Writer writer) throws TransformerException
   {
      Source xmlSource = new StreamSource(reader);
      Result xmlResult = new StreamResult(writer);         
      Source xslSource = new StreamSource(new StringReader(config.getXslt()));
      
      TransformerFactory factory = TransformerFactory.newInstance();
      Transformer transformer = factory.newTransformer(xslSource);

      transformer.setParameter("port", new Integer(binding.getPort()));
      String host = binding.getHostName();
      if (host != null)
      {
         transformer.setParameter("host", host);
      }

      // Check for any arbitrary attributes
      Map<String, String> attributes = config.getAdditionalAttributes();
      for(Map.Entry<String, String> entry : attributes.entrySet())
      {
         String attrValue = StringPropertyReplacer.replaceProperties(entry.getValue());
         transformer.setParameter(entry.getKey(), attrValue);
      }

      transformer.transform(xmlSource, xmlResult);
   }

}
