/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.weld.integration.deployer.metadata;

import java.util.Collection;
import java.util.Iterator;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.metadata.ejb.jboss.JBossAssemblyDescriptorMetaData;
import org.jboss.metadata.ejb.jboss.JBossMetaData;
import org.jboss.metadata.ejb.spec.AroundInvokeMetaData;
import org.jboss.metadata.ejb.spec.AroundInvokesMetaData;
import org.jboss.metadata.ejb.spec.EjbJar30MetaData;
import org.jboss.metadata.ejb.spec.EjbJar3xMetaData;
import org.jboss.metadata.ejb.spec.InterceptorBindingMetaData;
import org.jboss.metadata.ejb.spec.InterceptorBindingsMetaData;
import org.jboss.metadata.ejb.spec.InterceptorClassesMetaData;
import org.jboss.metadata.ejb.spec.InterceptorMetaData;
import org.jboss.metadata.ejb.spec.InterceptorsMetaData;
import org.jboss.vfs.VirtualFile;
import org.jboss.weld.integration.deployer.DeployersUtils;

/**
 * Adds wb custom interceptor to ejb deployments.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
@SuppressWarnings("deprecation")
public class WeldEjbInterceptorMetadataDeployer extends WeldAwareMetadataDeployer<JBossMetaData>
{
   public static final String INJECTION_INTERCEPTOR_CLASS_NAME = "org.jboss.weld.integration.ejb.SessionBeanInterceptor";
   public static final String CONTEXT_INTERCEPTOR_CLASS_NAME = "org.jboss.weld.ejb.SessionBeanInterceptor";
   public static final String BINDINGS_INTERCEPTOR_CLASS_NAME = "org.jboss.weld.integration.ejb.interceptor.Jsr299BindingsInterceptor";

   private InterceptorMetaData injectionIMD;
   private InterceptorMetaData bindingsIMD;
   private InterceptorMetaData contextIMD;
   private InterceptorBindingMetaData injectionIBMD;
   private InterceptorBindingMetaData bindingsIBMD;
   private InterceptorBindingMetaData contextIBMD;

   public WeldEjbInterceptorMetadataDeployer()
   {
      super(JBossMetaData.class, true);

      addInput("merged." + JBossMetaData.class.getName());
      setStage(DeploymentStages.POST_CLASSLOADER);

      // create interceptor metadata instance for session beans
      injectionIMD = new InterceptorMetaData();
      injectionIMD.setInterceptorClass(INJECTION_INTERCEPTOR_CLASS_NAME);

      contextIMD = new InterceptorMetaData();
      contextIMD.setInterceptorClass(CONTEXT_INTERCEPTOR_CLASS_NAME);

      if (contextIMD.getAroundInvokes() == null)
      {
         contextIMD.setAroundInvokes(new AroundInvokesMetaData());
      }
      AroundInvokeMetaData aroundInvokeMetaData = new AroundInvokeMetaData();
      aroundInvokeMetaData.setClassName(CONTEXT_INTERCEPTOR_CLASS_NAME);
      aroundInvokeMetaData.setMethodName("aroundInvoke");
      contextIMD.getAroundInvokes().add(aroundInvokeMetaData);

      // create interceptor metadata instance for JSR-299 specific bindings
      bindingsIMD = new InterceptorMetaData();
      bindingsIMD.setInterceptorClass(BINDINGS_INTERCEPTOR_CLASS_NAME);

      // create interceptor binding metadata instance
      injectionIBMD = createInterceptorBindingMetadata(INJECTION_INTERCEPTOR_CLASS_NAME);
      bindingsIBMD = createInterceptorBindingMetadata(BINDINGS_INTERCEPTOR_CLASS_NAME);
      contextIBMD = createInterceptorBindingMetadata(CONTEXT_INTERCEPTOR_CLASS_NAME);

   }

   private InterceptorBindingMetaData createInterceptorBindingMetadata(String interceptorClassName)
   {
      InterceptorBindingMetaData ibmd = new InterceptorBindingMetaData();
      InterceptorClassesMetaData interceptorClasses = new InterceptorClassesMetaData();
      interceptorClasses.add(interceptorClassName);
      ibmd.setInterceptorClasses(interceptorClasses);
      ibmd.setEjbName("*");
      return ibmd;
   }

   @Override
   protected void internalDeploy(VFSDeploymentUnit unit, JBossMetaData jbmd, Collection<VirtualFile> wbXml) throws DeploymentException
   {
      if (jbmd.getInterceptors() == null)
      {
         InterceptorsMetaData imd = new InterceptorsMetaData();
         EjbJar3xMetaData ejmd = new EjbJar30MetaData();
         ejmd.setInterceptors(imd);
         jbmd.merge(null, ejmd);
      }
      InterceptorsMetaData interceptors = jbmd.getInterceptors();
      interceptors.add(injectionIMD); // clone?
      interceptors.add(bindingsIMD);
      interceptors.add(contextIMD);

      JBossAssemblyDescriptorMetaData assemblyDescriptor = jbmd.getAssemblyDescriptor();
      if (assemblyDescriptor == null)
      {
         assemblyDescriptor = new JBossAssemblyDescriptorMetaData();
         jbmd.setAssemblyDescriptor(assemblyDescriptor);
      }
      InterceptorBindingsMetaData interceptorBindings = assemblyDescriptor.getInterceptorBindings();
      if (interceptorBindings == null)
      {
         interceptorBindings = new InterceptorBindingsMetaData();
         assemblyDescriptor.setInterceptorBindings(interceptorBindings);
      }
      // Add this as the first binding in the list so that it is called first...
      interceptorBindings.add(contextIBMD);
      interceptorBindings.add(0, injectionIBMD);

      // Add JSR-299 binding interceptor as the last (all others have to be
      // called first)
      interceptorBindings.add(bindingsIBMD);

      // Check to see there is a defined order; if we aren't first, warn
      for (InterceptorBindingMetaData interceptorBinding : interceptorBindings)
      {
         if (interceptorBinding.getInterceptorOrder() != null && !interceptorBinding.getInterceptorOrder().isEmpty())
         {
            if (!INJECTION_INTERCEPTOR_CLASS_NAME.equals(interceptorBinding.getInterceptorOrder().iterator().next()))
            {
               log.warn("The Web Beans SessionnBeanInterceptor is not the inner most EJB interceptor in this deployment. JSR299 injection may not work correctly. Specify " + INJECTION_INTERCEPTOR_CLASS_NAME + " as the first interceptor in the interceptor ordering for " + interceptorBinding.getEjbName());
            }
            // TODO automagically make ours the first?
            Object lastInterceptorClassName;
            Iterator<String> iterator = interceptorBinding.getInterceptorOrder().iterator();
            do
            {
               lastInterceptorClassName = iterator.next();
            }
            while (iterator.hasNext());
            if (!BINDINGS_INTERCEPTOR_CLASS_NAME.equals(lastInterceptorClassName))
            {
               log.warn("The Web Beans Jsr299BindingsInterceptor is not the outer most EJB interceptor in this deployment. JSR299 requires that @InterceptorBinding-bound interceptors should execute after @Interceptors-registered interceptors. Specify " + BINDINGS_INTERCEPTOR_CLASS_NAME + " as the last interceptor in the interceptor ordering for " + interceptorBinding.getEjbName());
            }
         }
      }
   }
}