/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.jsf.integration.config;

import java.util.Map;
import javax.faces.validator.BeanValidator;
import javax.servlet.ServletContext;
import javax.servlet.ServletRegistration;
import javax.validation.Validation;
import javax.validation.ValidatorFactory;
import org.jboss.logging.Logger;
import org.jboss.mc.servlet.vdf.spi.VDFConnector;
import org.jboss.web.validation.ValidatorFactoryVDFConnector;

/**
 * This utility class is used to do all JBoss-specific initialization for JSF.
 *
 * @author ssilvert
 * @author Ales Justin
 */
public class JBossJSFInitializer {

   private static Logger LOG = Logger.getLogger(JBossJSFInitializer.class);

   private ServletContext servletContext;

   /**
    * Create a new JBossJSFInitializer.
    *
    * @param servletContext The ServletContext object for this WAR deployemnt.
    */
   public JBossJSFInitializer(ServletContext servletContext)
   {
      this.servletContext = servletContext;
   }

   /**
    * Adds the BeanValidatorFacotry to JSF2 implementations.
    */
   public void addBeanValidatorFactory()
   {
       VDFConnector<ValidatorFactory> vfc = new ValidatorFactoryVDFConnector(servletContext);

       ValidatorFactory factory;
       if (vfc.isValid())
       {
          factory = vfc.getUtility();
       }
       else
       {
          LOG.warn("No such ValidatorFactory in VDF layer, creating new instance.");
          factory = Validation.buildDefaultValidatorFactory();
       }

       servletContext.setAttribute(BeanValidator.VALIDATOR_FACTORY_KEY, factory);
    }

   /**
    * If no JSF servlet mapping exists, add the default.
    *
    * @return <code>Boolean.TRUE</code> if mappings were added,
    *         <code>Boolean.FALSE</code> otherwise.
    */
    public Boolean addFacesMappings()
    {
       if (hasJSFServlet()) return Boolean.FALSE;

       ServletRegistration.Dynamic facesServlet =
                  servletContext.addServlet("FacesServlet",
                                            "javax.faces.webapp.FacesServlet");
       facesServlet.addMapping("/faces/*", "*.jsf", "*.faces");

       return Boolean.TRUE;
    }

    private boolean hasJSFServlet()
    {
       Map<String,? extends ServletRegistration> servletRegistrations = servletContext.getServletRegistrations();
       for (ServletRegistration registration : servletRegistrations.values()) {
          // TODO: get other FacesServlet classes from the JSF deployer
          if ("javax.faces.webapp.FacesServlet".equals(registration.getClassName())) return true;
       }

       return false;
    }

    /**
     * This method accounts for a peculiar problem with Jasper that pops up from time
     * to time.  In some cases, if the JspRuntimeContext is not loaded then the JspFactory
     * will not be initialized for JSF.  This method assures that it will always be
     * be loaded before JSF is initialized.
     */
    public static void initializeJspRuntime()
    {
        try
        {
            Class.forName("org.apache.jasper.compiler.JspRuntimeContext");
        }
        catch (ClassNotFoundException cnfe)
        {
            // do nothing
        }
    }
}
