package org.jboss.jsr299.tck;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jboss.jsr299.tck.api.DeploymentException;
import org.jboss.jsr299.tck.api.TestResult;
import org.jboss.jsr299.tck.api.TestResult.Status;
import org.jboss.jsr299.tck.impl.packaging.ArtifactGenerator;
import org.jboss.jsr299.tck.impl.packaging.jsr299.TCKArtifactDescriptor;
import org.testng.IHookCallBack;
import org.testng.IHookable;
import org.testng.ITestContext;
import org.testng.ITestResult;
import org.testng.annotations.AfterClass;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.AfterSuite;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.BeforeSuite;

public abstract class AbstractDeclarativeTest extends AbstractTest implements IHookable
{
   
   private static Logger log = Logger.getLogger(AbstractDeclarativeTest.class);

   private static boolean inContainer = false;
   
   public static boolean isInContainer()
   {
      return inContainer;
   }
   
   public static void setInContainer(boolean inContainer)
   {
      AbstractDeclarativeTest.inContainer = inContainer;
   }

   private TCKArtifactDescriptor artifact;
   private DeploymentException deploymentException;
   private boolean skipTest = false;
   
   private boolean isSuiteDeployingTestsToContainer()
   {
      return !isInContainer() && (!getCurrentConfiguration().isStandalone() || getCurrentConfiguration().isRunIntegrationTests()); 
   }
   
   private void generateArtifact()
   {
      // If we are in the container, the artifact is already built
      if (!isInContainer())
      {
         ArtifactGenerator generator = new ArtifactGenerator(getCurrentConfiguration());
         artifact = generator.createArtifact(this.getClass());
      }
   }
   
   private boolean isDeployToContainerNeeded()
   {
      /*
       *    If this isn't running inside the container
       *  AND
       *    there is an artifact to deploy
       *  AND
       *    EITHER
       *      we are in standalone mode and it isn't a unit test
       *    OR
       *      we aren't in standalone mode
       *  THEN
       *    we need to deploy 
       */
      return !isInContainer() && artifact != null && ((getCurrentConfiguration().isStandalone() && !artifact.isUnit() && getCurrentConfiguration().isRunIntegrationTests()) || !getCurrentConfiguration().isStandalone());
   }
   
   private void deployArtifact()
   {
      try
      {
         if (isDeployToContainerNeeded())
         {
            InputStream jar = null;
            try
            {
               jar = artifact.getJarAsStream();
               getCurrentConfiguration().getContainers().deploy(jar, artifact.getDefaultName());
            }
            finally
            {
               if (jar != null)
               {
                  jar.close();
               }
            }
         }
         else if (artifact != null && artifact.isUnit())
         {
            Set<Class<?>> classes = artifact.getClasses();
            getCurrentConfiguration().getStandaloneContainers().deploy(classes, Arrays.asList(artifact.getBeansXml().getSource()));
         }
      }
      catch (IOException e)
      {
         throw new RuntimeException("Error connecting to the container", e);
      }
      catch (DeploymentException e)
      {
         deploymentException = e;
      }
      if (artifact != null && artifact.getExpectedDeploymentException() != null)
      {
         if (deploymentException != null)
         {
            if (isThrowablePresent(artifact.getExpectedDeploymentException(), deploymentException.getCause()))
            {
               // We expect this exception, so ignore it
               deploymentException = null;
               skipTest = true;
            }
         }
         else
         {
            this.deploymentException = new DeploymentException(new ExpectedException("Expected exception " + artifact.getExpectedDeploymentException() + " but none was thrown"));
         }
      }
   }
   
   private void undeployArtifact() throws Exception
   {
      if (isDeployToContainerNeeded())
      {
         getCurrentConfiguration().getContainers().undeploy(artifact.getDefaultName());
      }
      if (getCurrentConfiguration().isStandalone() && artifact != null && artifact.isUnit())
      {
         getCurrentConfiguration().getStandaloneContainers().undeploy();
      }
   }
   
   private void checkAssertionsEnabled()
   {
      boolean assertionsEnabled = false;
      try
      {
         assert false;
      }
      catch (AssertionError error) 
      {
         assertionsEnabled = true;
      }
      if (!assertionsEnabled)
      {
         throw new IllegalStateException("Assertions must be enabled!");
      }
   }
   
   @BeforeSuite(alwaysRun=true)
   public void beforeSuite(ITestContext context) throws Exception
   {
      if (isSuiteDeployingTestsToContainer())
      {
         getCurrentConfiguration().getContainers().setup();
      }
      if (getCurrentConfiguration().isStandalone())
      {
         getCurrentConfiguration().getStandaloneContainers().setup();
      }
      checkAssertionsEnabled();
   }
   
   
   @AfterSuite(alwaysRun=true)
   public void afterSuite() throws Exception
   {
      if (isSuiteDeployingTestsToContainer())
      {
         getCurrentConfiguration().getContainers().cleanup();
      }
      if (getCurrentConfiguration().isStandalone())
      {
         getCurrentConfiguration().getStandaloneContainers().cleanup();
      }
   }
   
   @BeforeClass(alwaysRun=true)
   public void beforeClass() throws Throwable
   {
      generateArtifact();
      deployArtifact();
      
   }
   
   @AfterClass(alwaysRun=true)
   public void afterClass() throws Exception
   {
      undeployArtifact();
      this.artifact = null;
      this.deploymentException = null;
      skipTest = false;
   }

   @BeforeMethod(alwaysRun=true)
   public void beforeMethod(Method method)
   {
      setCurrentManager(getCurrentConfiguration().getManagers().getManager());
   }

   @AfterMethod(alwaysRun=true)
   public void afterMethod()
   {
      setCurrentManager(null);
   }


   public void run(IHookCallBack callback, ITestResult testResult)
   {
      if (artifact == null && !isInContainer())
      {
         log.warn("Non @Artifact-test for testcase " + testResult.getMethod());
      }
	   if (deploymentException != null)
	   {
		   testResult.setThrowable(deploymentException.getCause());
	   }
	   else if ((!isDeployToContainerNeeded() || artifact.isRunLocally())  && !skipTest)
      {
         callback.runTestMethod(testResult);
         if (!getCurrentConfiguration().isStandalone() && !isInContainer() && !artifact.isRunLocally())
         {
            log.warn("Running testcase locally " + testResult.getMethod());
         }
      }
      else if (!skipTest)
      {
         try
         {
            TestResult result = getCurrentConfiguration().getInContainerTestLauncher().launchTest(testResult.getMethod().getMethod());
            if (result.getStatus().equals(Status.FAILED) || result.getStatus().equals(Status.SKIPPED))
            {
               testResult.setThrowable(result.getThrowable());
               testResult.setStatus(ITestResult.FAILURE);
            }
         }
         catch (IOException e)
         {
            throw new RuntimeException("Error connecting to the container", e);
         }
      }
   }
}