package org.jboss.jsr299.tck.impl.packaging;

import static org.jboss.jsr299.tck.impl.packaging.PackagingType.EAR;
import static org.jboss.jsr299.tck.impl.packaging.PackagingType.WAR;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jboss.jsr299.tck.AbstractTest;
import org.jboss.jsr299.tck.api.Configuration;
import org.jboss.jsr299.tck.impl.packaging.ear.EarArtifactDescriptor;
import org.jboss.jsr299.tck.impl.packaging.jsr299.BeansXml;
import org.jboss.jsr299.tck.impl.packaging.jsr299.TCKArtifactDescriptor;
import org.jboss.jsr299.tck.impl.packaging.war.WarArtifactDescriptor;

public class ArtifactGenerator
{
   
   private static class ArtifactProcessor
   {
      
      private static final Logger log = Logger.getLogger(ArtifactProcessor.class);
      
      private final boolean unit;
      private final boolean runLocally;
      private final boolean addDeclaringPackage;
      private final String beansXml;
      private final PackagingType packagingType;
      private final Collection<ResourceDescriptor> resources;
      private final Collection<Class<?>> classes;
      private final Class<? extends Throwable> expectedDeploymentException;
      private final Set<ResourceDescriptor> extraLibraries;
      
      
      private final Class<?> declaringClass;
      private final boolean standalone;

      public ArtifactProcessor(Class<?> declaringClass, boolean standalone, String extraLibrariesDirectory)
      {
         this.standalone = standalone;
         this.declaringClass = declaringClass;
         
         if (declaringClass.isAnnotationPresent(Artifact.class))
         {
            this.addDeclaringPackage = declaringClass.getAnnotation(Artifact.class).addCurrentPackage();
         }
         else
         {
            throw new IllegalStateException("Unable to find @Artifact on " + declaringClass);
         }
         
         if (declaringClass.isAnnotationPresent(BeansXml.class))
         {
            this.beansXml = asAbsolutePath(declaringClass.getAnnotation(BeansXml.class).value());
         }
         else
         {
            this.beansXml = null;
         }
         
         if (declaringClass.isAnnotationPresent(Packaging.class))
         {
            this.packagingType = declaringClass.getAnnotation(Packaging.class).value();
         }
         else
         {
            this.packagingType = WAR;
         }
         
         if (declaringClass.isAnnotationPresent(IntegrationTest.class))
         {
            this.unit = false;
            this.runLocally = declaringClass.getAnnotation(IntegrationTest.class).runLocally(); 
         }
         else
         {
            this.unit = true;
            this.runLocally = false;
         }
         
         if (declaringClass.isAnnotationPresent(Resources.class))
         {
            this.resources = asResourceDescriptors(declaringClass.getAnnotation(Resources.class).value());
         }
         else
         {
            this.resources = Collections.emptyList();
         }
         
         if (declaringClass.isAnnotationPresent(Classes.class))
         {
            this.classes = Arrays.asList(declaringClass.getAnnotation(Classes.class).value());
         }
         else
         {
            this.classes = Collections.emptyList();
         }
         if (declaringClass.isAnnotationPresent(ExpectedDeploymentException.class))
         {
            this.expectedDeploymentException = declaringClass.getAnnotation(ExpectedDeploymentException.class).value();
         }
         else
         {
            this.expectedDeploymentException = null;
         }
         if (extraLibrariesDirectory != null)
         {
            File directory = new File(extraLibrariesDirectory);
            this.extraLibraries = new HashSet<ResourceDescriptor>();
            if (directory.isDirectory())
            {
               for (File file : directory.listFiles(new FilenameFilter()
                  {
                     
                     public boolean accept(File dir, String name)
                     {
                        return name.endsWith(".jar");
                     }
                     
                  }))
               {
                  try
                  {
                     this.extraLibraries.add(new ResourceDescriptor(file.getName(), file.toURI().toURL()));
                  }
                  catch (IOException e)
                  {
                     log.warn("Unable to load extra library", e);
                  }
               }
            }
         }
         else
         {
            this.extraLibraries = Collections.emptySet();
         }
      }
      
      public TCKArtifactDescriptor createArtifact()
      {
         final TCKArtifactDescriptor artifact = newArtifact(packagingType, declaringClass, beansXml, standalone, addDeclaringPackage);
         artifact.setUnit(unit);
         artifact.setRunLocally(runLocally);
         artifact.setExpectedDeploymentException(expectedDeploymentException);
         artifact.getClasses().addAll(classes);
         // Annoying hack
         artifact.getResources().removeAll(resources);
         artifact.getResources().addAll(resources);
         artifact.getLibraries().addAll(extraLibraries);
         return artifact;
      }
      
      private Collection<ResourceDescriptor> asResourceDescriptors(Resource[] resources)
      {
         List<ResourceDescriptor> resourceDescriptors = new ArrayList<ResourceDescriptor>();
         for (Resource resource : resources)
         {
            resourceDescriptors.add(new ResourceDescriptor(resource.destination(), asAbsolutePath(resource.source())));
         }
         return resourceDescriptors;
      }
      
      private String asAbsolutePath(String path)
      {
         if (path.startsWith("/"))
         {
            return path.substring(1);
         }
         else
         {
            return declaringClass.getPackage().getName().replace(".", "/") + "/" + path;
         }
      }
      
      private static TCKArtifactDescriptor newArtifact(PackagingType packagingType, Class<?> declaringClass, String beansXml, boolean standalone, boolean addDeclaringPackage)
      {
         TCKArtifactDescriptor artifact;
         if (!standalone && packagingType.equals(WAR))
         {
               artifact = new WarArtifactDescriptor(declaringClass, beansXml);
         } 
         else if (!standalone && packagingType.equals(EAR))
         {
            artifact = new EarArtifactDescriptor(declaringClass, beansXml);
         }
         else
         {
            artifact = new TCKArtifactDescriptor(declaringClass, beansXml);
         }
         
         if (addDeclaringPackage)
         {
            artifact.addPackage(declaringClass.getPackage());
         }
         return artifact;
      }
      
      public boolean isUnit()
      {
         return unit;
      };
      
   }
   
   private static final Logger log = Logger.getLogger(ArtifactGenerator.class);
   
   private final Configuration configuration;
   
   public ArtifactGenerator(Configuration configuration)
   {
      if (configuration == null)
      {
         throw new IllegalArgumentException("configuration must not be null");
      }
      this.configuration = configuration;
   }

   public List<ArtifactDescriptor> createArtifacts(String packageName)
   {
      Set<Class<?>> classes = new HashSet<Class<?>>();
      classes.addAll(new ArtifactScanner(packageName, null).getClasses());
      List<ArtifactDescriptor> artifacts = new ArrayList<ArtifactDescriptor>();
      for (Class<?> clazz : classes)
      {
         artifacts.add(createArtifact(clazz));
      }
      return artifacts;
   }
   
   public void dumpArtifacts() 
   {
      List<ArtifactDescriptor> artifacts = createArtifacts(AbstractTest.class.getPackage().getName());
      File file = new File(configuration.getOutputDirectory());
      if (!file.exists())
      {
         file.mkdirs();
      }
      else if (file.isFile())
      {
         throw new IllegalStateException("Cannot use debug directory " + configuration.getOutputDirectory() + ", it already exists");
      }
      log.info("Writing artifacts to " + configuration.getOutputDirectory());
      for (ArtifactDescriptor artifact : artifacts)
      {
         try
         {
            artifact.writeArtifactToDisk(configuration.getOutputDirectory());
            log.info("Written artifact to disk " + artifact);
         }
         catch (IOException e)
         {
            log.warn("Error writing artifact to disk " + artifact, e);
         }
      }
   }

   public TCKArtifactDescriptor createArtifact(Class<?> declaringClass)
   {
      if (declaringClass.isAnnotationPresent(Artifact.class))
      {
         return new ArtifactProcessor(declaringClass, configuration.isStandalone(), configuration.getLibraryDirectory()).createArtifact();
      }
      else
      {
         return null;
      }
   }
   
}
