package org.jboss.jsr299.tck.tests.context.dependent;

import java.lang.annotation.Annotation;
import java.util.Set;

import javax.context.Context;
import javax.context.ContextNotActiveException;
import javax.context.Dependent;
import javax.inject.AnnotationLiteral;
import javax.inject.manager.Bean;

import org.hibernate.tck.annotations.SpecAssertion;
import org.hibernate.tck.annotations.SpecAssertions;
import org.jboss.jsr299.tck.AbstractDeclarativeTest;
import org.jboss.jsr299.tck.impl.packaging.Artifact;
import org.jboss.jsr299.tck.impl.packaging.jsr299.BeansXml;
import org.testng.annotations.Test;

/**
 * Spec version: Public Release Draft 2
 */
@Artifact
@BeansXml("beans.xml")
public class DependentContextTest extends AbstractDeclarativeTest
{
   
   private static final Annotation TAME_LITERAL = new AnnotationLiteral<Tame> () {};
   
   @Test(groups = { "contexts", "injection" })
   @SpecAssertion(section = "8.3", id = "a")
   public void testInstanceNotSharedBetweenInjectionPoints()
   {
      Set<Bean<Fox>> foxBeans = getCurrentManager().resolveByType(Fox.class);
      assert foxBeans.size() == 1;
      Set<Bean<FoxRun>> foxRunBeans = getCurrentManager().resolveByType(FoxRun.class);
      assert foxRunBeans.size() == 1;
      Bean<FoxRun> foxRunBean = foxRunBeans.iterator().next();
      FoxRun foxRun = foxRunBean.create(new MockCreationalContext<FoxRun>());
      assert !foxRun.fox.equals(foxRun.anotherFox);
   }

   @Test(groups = { "contexts", "el" })
   @SpecAssertion(section = "8.3", id = "c")
   public void testInstanceUsedForElEvaluationNotShared() throws Exception
   {
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            Set<Bean<Fox>> foxBeans = getCurrentManager().resolveByType(Fox.class);
            assert foxBeans.size() == 1;

            Fox fox1 = getCurrentConfiguration().getEl().evaluateValueExpression("#{fox}", Fox.class);
            Fox fox2 = getCurrentConfiguration().getEl().evaluateValueExpression("#{fox}", Fox.class);
            assert !fox1.equals(fox2);
         }
      }.run();
   }

   @Test(groups = { "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3", id = "d")
   public void testInstanceUsedForProducerMethodNotShared() throws Exception
   {
      Bean<Tarantula> tarantulaBean = getCurrentManager().resolveByType(Tarantula.class).iterator().next();
      Tarantula tarantula = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      Tarantula tarantula2 = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      assert tarantula != null;
      assert tarantula2 != null;
      assert tarantula != tarantula2;
   }

   @Test(groups = { "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3", id = "d")
   public void testInstanceUsedForProducerFieldNotShared() throws Exception
   {
      Bean<Tarantula> tarantulaBean = getCurrentManager().resolveByType(Tarantula.class, TAME_LITERAL).iterator().next();
      Tarantula tarantula = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      Tarantula tarantula2 = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      assert tarantula != null;
      assert tarantula2 != null;
      assert tarantula != tarantula2;
   }

   @Test(groups = { "broken", "contexts", "disposalMethod" })
   @SpecAssertion(section = "8.3", id = "d")
   public void testInstanceUsedForDisposalMethodNotShared() throws Exception
   {
//      deployBeans(SpiderProducer.class);
//      new RunInDependentContext()
//      {
//
//         @Override
//         protected void execute() throws Exception
//         {
//            SpiderProducer spiderProducer = getCurrentManager().getInstanceByType(SpiderProducer.class);
//            Bean<Tarantula> tarantulaBean = getCurrentManager().resolveByType(Tarantula.class).iterator().next();
//            Tarantula tarantula = tarantulaBean.create(new MyCreationalContext<Tarantula>());
//            assert tarantula != null;
//            tarantulaBean.destroy(tarantula);
//            assert SpiderProducer.getInstanceUsedForDisposal() != null;
//            assert SpiderProducer.getInstanceUsedForDisposal() != spiderProducer;
//         }
//
//      }.run();
      assert false;
   }

   @Test(groups = { "contexts", "observerMethod" })
   @SpecAssertion(section = "8.3", id = "d")
   public void testInstanceUsedForObserverMethodNotShared() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            HorseStable firstStableInstance = getCurrentManager().getInstanceByType(HorseStable.class);
            getCurrentManager().fireEvent(new HorseInStableEvent());
            assert HorseStable.getInstanceThatObservedEvent() != null;
            assert HorseStable.getInstanceThatObservedEvent() != firstStableInstance;
         }

      }.run();
   }

   @Test(groups = "contexts")
   @SpecAssertion(section = "8.3", id = "e")
   public void testContextGetWithCreateTrueReturnsNewInstance() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            Set<Bean<Fox>> foxBeans = getCurrentManager().resolveByType(Fox.class);
            assert foxBeans.size() == 1;
            Bean<Fox> foxBean = foxBeans.iterator().next();
            Context context = getCurrentManager().getContext(Dependent.class);
            assert context.get(foxBean, new MockCreationalContext<Fox>()) != null;
            assert context.get(foxBean, new MockCreationalContext<Fox>()) instanceof Fox;
         }

      }.run();
   }

   @Test(groups = "contexts")
   @SpecAssertion(section = "8.3", id = "f")
   public void testContextGetWithCreateFalseReturnsNull() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            Set<Bean<Fox>> foxBeans = getCurrentManager().resolveByType(Fox.class);
            assert foxBeans.size() == 1;
            Bean<Fox> foxBean = foxBeans.iterator().next();
            Context context = getCurrentManager().getContext(Dependent.class);
            assert context.get(foxBean, null) == null;
         }

      }.run();
   }

   @Test(groups = { "contexts", "ri-broken" }, expectedExceptions = ContextNotActiveException.class)
   @SpecAssertion(section = "8.3", id = "g")
   public void testContextIsInactive()
   {
      // The RI test harness is broken here, it just enables the dependent context blindly
      getCurrentManager().getContext(Dependent.class);
   }

   @Test(groups = { "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3", id = "g")
   public void testContextIsActiveWhenInvokingProducerMethod()
   {
      Bean<Tarantula> tarantulaBean = getCurrentManager().resolveByType(Tarantula.class).iterator().next();
      Tarantula tarantula = tarantulaBean.create(new MockCreationalContext<Tarantula>());
      assert tarantula != null;
      assert SpiderProducer.isDependentContextActive();
   }

   @Test(groups = { "contexts", "producerField"})
   @SpecAssertion(section = "8.3", id = "g")
   public void testContextIsActiveWhenInvokingProducerField()
   {
      // Reset test class
      Tarantula.setDependentContextActive(false);
      getCurrentManager().getInstanceByType(Tarantula.class, TAME_LITERAL);
      assert Tarantula.isDependentContextActive();
   }

   @Test(groups = { "broken", "contexts", "disposalMethod" })
   @SpecAssertions({
      @SpecAssertion(section = "8.3", id = "g"),
      @SpecAssertion(section = "8.3", id = "j")
   })
   public void testContextIsActiveWhenInvokingDisposalMethod()
   {
//      deployBeans(SpiderProducer.class);
//      Bean<Tarantula> tarantulaBean = getCurrentManager().resolveByType(Tarantula.class).iterator().next();
//      Tarantula tarantula = tarantulaBean.create(new MyCreationalContext<Tarantula>());
//      assert tarantula != null;
//      SpiderProducer.setDependentContextActive(false);
//      tarantulaBean.destroy(tarantula);
//      assert SpiderProducer.isDependentContextActive();
      assert false;
   }

   @Test(groups = { "contexts", "observerMethod" })
   @SpecAssertion(section = "8.3", id = "g")
   public void testContextIsActiveWhenCreatingObserverMethodInstance()
   {
      getCurrentManager().fireEvent(new HorseInStableEvent());
      assert HorseStable.isDependentContextActive();
   }

   @Test(groups = { "contexts", "el" })
   @SpecAssertion(section = "8.3", id = "h")
   public void testContextIsActiveWhenEvaluatingElExpression() throws Exception
   {
      SensitiveFox.setManager(getCurrentManager());
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            String foxName = getCurrentConfiguration().getEl().evaluateMethodExpression("#{sensitiveFox.getName}", String.class, new Class[0], new Object[0]);
            assert foxName != null;
            assert SensitiveFox.isDependentContextActiveDuringEval();
         }
      }.run();
   }

   @Test(groups = { "contexts", "observerMethod" })
   @SpecAssertion(section = "8.3", id = "i")
   public void testContextIsActiveWhenInvokingObserverMethod()
   {
      getCurrentManager().fireEvent(new HorseInStableEvent());
      assert ApplicationHorseStable.isDependentContextActive();
   }

   @Test(groups = { "contexts", "beanLifecycle" })
   @SpecAssertion(section = "8.3", id = "j")
   public void testContextIsActiveDuringBeanCreation() throws Exception
   {
      SensitiveFox.setManager(getCurrentManager());
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            SensitiveFox fox1 = getCurrentManager().getInstanceByType(SensitiveFox.class);
            assert fox1 != null;
            assert fox1.isDependentContextActiveDuringCreate();
         }

      }.run();
   }

   @Test(groups = { "contexts", "injection" })
   @SpecAssertion(section = "8.3", id = "j")
   public void testContextIsActiveDuringInjection()
   {
      Bean<FoxRun> foxRunBean = getCurrentManager().resolveByType(FoxRun.class).iterator().next();
      FoxRun foxRun = foxRunBean.create(new MockCreationalContext<FoxRun>());
      assert foxRun.fox != null;
   }

   @Test(groups = { "contexts", "beanDestruction"})
   @SpecAssertions({
      @SpecAssertion(section = "8.3.2", id = "a"),
      @SpecAssertion(section = "8.3", id = "b")
   })
   public void testDestroyingSimpleParentDestroysDependents() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            assert getCurrentManager().resolveByType(Farm.class).size() == 1;
            Bean<Farm> farmBean = getCurrentManager().resolveByType(Farm.class).iterator().next();
            Farm farm = getCurrentManager().getInstanceByType(Farm.class);
            Stable.destroyed = false;
            Horse.destroyed = false;
            farmBean.destroy(farm);
            assert Stable.destroyed;
            assert Horse.destroyed;
         }

      }.run();
   }
   
   @Test(groups = { "contexts", "beanDestruction"})
   @SpecAssertions({
      @SpecAssertion(section = "8.3.2", id = "a"),
      @SpecAssertion(section = "8.3", id = "b")
   })
   public void testDestroyingSimpleParentDestroysDependentsOfSameBean() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            // Reset test class
            Fox.setDestroyed(false);
            Fox.setDestroyCount(0);
            
            assert getCurrentManager().resolveByType(FoxRun.class).size() == 1;
            Bean<FoxRun> bean = getCurrentManager().resolveByType(FoxRun.class).iterator().next();
            FoxRun instance = getCurrentManager().getInstanceByType(FoxRun.class);
            assert instance.fox != instance.anotherFox;
            bean.destroy(instance);
            assert Fox.isDestroyed();
            assert Fox.getDestroyCount() == 2;
         }

      }.run();
   }

   @Test(groups = { "contexts", "el"})
   @SpecAssertion(section = "8.3.2", id = "c")
   public void testDependentsDestroyedWhenElEvaluationCompletes() throws Exception
   {
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            // Reset test class
            Fox.setDestroyed(false);
            FoxRun.setDestroyed(false);
            
            getCurrentConfiguration().getEl().evaluateValueExpression("#{foxRun}", FoxRun.class);
            assert FoxRun.isDestroyed();
            assert Fox.isDestroyed();
         }
      }.run();
   }

   @Test(groups = { "contexts", "producerMethod" })
   @SpecAssertion(section = "8.3.2", id = "da")
   public void testDependentsDestroyedWhenProducerMethodCompletes() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            // Reset the test class
            SpiderProducer.setDestroyed(false);
            Tarantula spiderInstance = getCurrentManager().getInstanceByType(Tarantula.class);
            assert spiderInstance != null;
            assert SpiderProducer.isDestroyed();
         }

      }.run();
   }

   @Test(groups = { "contexts", "producerField" })
   @SpecAssertion(section = "8.3.2", id = "db")
   public void testDependentsDestroyedWhenProducerFieldCompletes() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            // Reset the test class
            OtherSpiderProducer.setDestroyed(false);
            
            Tarantula spiderInstance = getCurrentManager().getInstanceByType(Tarantula.class, TAME_LITERAL);
            assert spiderInstance != null;
            assert OtherSpiderProducer.isDestroyed();
            
         }

      }.run();
   }

   @Test(groups = { "contexts", "disposalMethod", "ri-broken" })
   @SpecAssertion(section = "8.3.2", id = "dc")
   public void testDependentsDestroyedWhenDisposalMethodCompletes() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            Bean<Tarantula> tarantulaBean = getCurrentManager().resolveByType(Tarantula.class).iterator().next();
            Tarantula tarantula = tarantulaBean.create(new MockCreationalContext<Tarantula>());
            assert tarantula != null;
            
            // Reset test class state
            SpiderProducer.setDestroyed(false);
            
            tarantulaBean.destroy(tarantula);
            assert SpiderProducer.isDestroyed();
         }

      }.run();
   }

   @Test(groups = { "contexts", "observerMethod" })
   @SpecAssertion(section = "8.3.2", id = "dd")
   public void testDependentsDestroyedWhenObserverMethodEvaluationCompletes() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            // Reset test class state...
            HorseStable.setInstanceThatObservedEvent(null);
            HorseStable.setDestroyed(false);
            
            getCurrentManager().fireEvent(new HorseInStableEvent());
            assert HorseStable.getInstanceThatObservedEvent() != null;
            assert HorseStable.isDestroyed();
         }

      }.run();
   }

}
