/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,  
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.jsr299.tck.tests.lookup.injectionpoint;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import javax.context.Dependent;
import javax.inject.Current;
import javax.inject.Standard;
import javax.inject.manager.Bean;
import javax.inject.manager.InjectionPoint;

import org.hibernate.tck.annotations.SpecAssertion;
import org.hibernate.tck.annotations.SpecAssertions;
import org.jboss.jsr299.tck.AbstractDeclarativeTest;
import org.jboss.jsr299.tck.impl.packaging.Artifact;
import org.jboss.jsr299.tck.literals.CurrentBinding;
import org.testng.annotations.Test;

/**
 * Injection point metadata tests
 * 
 * @author David Allen
 * 
 * Spec version: PRD2
 */
@Artifact
public class InjectionPointTest extends AbstractDeclarativeTest
{

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6", id = "a")
   public void testGetBean() throws Exception
   {

      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            FieldInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
   
            Set<Bean<FieldInjectionPointBean>> resolvedBeans = getCurrentManager().resolveByType(FieldInjectionPointBean.class);
            assert resolvedBeans.size() == 1;
            assert beanWithInjectionPoint.getInjectedMetadata().getBean().equals(resolvedBeans.iterator().next());
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6", id = "ba")
   public void testGetType() throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            FieldInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            assert beanWithInjectionPoint.getInjectedMetadata().getType().equals(BeanWithInjectionPointMetadata.class);
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6", id = "bc")
   public void testGetBindingTypes() throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            FieldInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            Set<Annotation> bindingTypes = beanWithInjectionPoint.getInjectedMetadata().getBindings();
            assert bindingTypes.size() == 1;
            assert Current.class.isAssignableFrom(bindingTypes.iterator().next().annotationType());
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6", id = "ca")
   public void testGetMemberField() throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            FieldInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            assert Field.class.isAssignableFrom(beanWithInjectionPoint.getInjectedMetadata().getMember().getClass());
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6", id = "cb")
   public void testGetMemberMethod() throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            MethodInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(MethodInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            assert Method.class.isAssignableFrom(beanWithInjectionPoint.getInjectedMetadata().getMember().getClass());
   
            // Since the type and bindings must correspond to the parameter, check
            // them
            assert beanWithInjectionPoint.getInjectedMetadata().getType().equals(BeanWithInjectionPointMetadata.class);
            assert beanWithInjectionPoint.getInjectedMetadata().getBindings().contains(new CurrentBinding());
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6", id = "cc")
   public void testGetMemberConstructor() throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            ConstructorInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(ConstructorInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            assert Constructor.class.isAssignableFrom(beanWithInjectionPoint.getInjectedMetadata().getMember().getClass());
   
            // Since the type and bindings must correspond to the parameter, check
            // them
            assert beanWithInjectionPoint.getInjectedMetadata().getType().equals(BeanWithInjectionPointMetadata.class);
            assert beanWithInjectionPoint.getInjectedMetadata().getBindings().contains(new CurrentBinding());
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6", id = "da")
   public void testGetAnnotation() throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            FieldInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            assert beanWithInjectionPoint.getInjectedMetadata().getAnnotation(AnimalStereotype.class) != null;
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6", id = "dd")
   public void testGetAnnotations() throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            FieldInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            Set<Annotation> annotations = new HashSet<Annotation>(Arrays.asList(beanWithInjectionPoint.getInjectedMetadata().getAnnotations()));
            assert annotations.size() > 0;
            assert annotations.contains(new CurrentBinding());
            assert annotations.contains(new AnimalStereotypeAnnotationLiteral());
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertions({
      @SpecAssertion(section = "5.6.1", id = "za"),
      @SpecAssertion(section = "2.5.1", id="a")
   })
   public void testStandardDeployment() throws Exception
   {
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            assert getCurrentManager().resolveByType(InjectionPoint.class).size() == 1;
            assert getCurrentManager().resolveByType(InjectionPoint.class).iterator().next().getDeploymentType().equals(Standard.class);
         }
         
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertion(section = "5.6.1", id = "zb")
   public void testDependentScope() throws Exception
   {
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            assert getCurrentManager().resolveByType(InjectionPoint.class).size() == 1;
            assert getCurrentManager().resolveByType(InjectionPoint.class).iterator().next().getScopeType().equals(Dependent.class);
         }
         
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertions({
      @SpecAssertion(section = "5.6.1", id = "zc")
   })
   public void testApiTypeInjectionPoint()  throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         
         @Override
         protected void execute() throws Exception
         {
            FieldInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            assert InjectionPoint.class.isAssignableFrom(beanWithInjectionPoint.getInjectedMetadata().getClass());
         }
      }.run();
   }

   @Test(groups = { "injectionPoint" })
   @SpecAssertions({
      @SpecAssertion(section = "5.6.1", id = "zd"),
      @SpecAssertion(section = "5.6.1", id = "a")
   })
   public void testCurrentBinding() throws Exception
   {
      // Get an instance of the bean which has another bean injected into it
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            FieldInjectionPointBean beanWithInjectedBean = getCurrentManager().getInstanceByType(FieldInjectionPointBean.class, new CurrentBinding());
            BeanWithInjectionPointMetadata beanWithInjectionPoint = beanWithInjectedBean.getInjectedBean();
            assert beanWithInjectionPoint.getInjectedMetadata() != null;
            assert beanWithInjectionPoint.getInjectedMetadata().getBindings().contains(new CurrentBinding());
         }
      }.run();
   }
}
