package org.jboss.jsr299.tck.tests.lookup.typesafe.resolution;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.inject.AnnotationLiteral;
import javax.inject.DefinitionException;
import javax.inject.DuplicateBindingTypeException;
import javax.inject.TypeLiteral;
import javax.inject.manager.Bean;

import org.hibernate.tck.annotations.SpecAssertion;
import org.hibernate.tck.annotations.SpecAssertions;
import org.jboss.jsr299.tck.AbstractDeclarativeTest;
import org.jboss.jsr299.tck.impl.packaging.Artifact;
import org.jboss.jsr299.tck.impl.packaging.jsr299.BeansXml;
import org.jboss.jsr299.tck.literals.CurrentBinding;
import org.testng.annotations.Test;

/**
 * 
 * Spec version: PRD2
 *
 */
@Artifact
@BeansXml("beans.xml")
public class ResolutionByTypeTest extends AbstractDeclarativeTest
{
   
   @Test(groups="resolution") 
   @SpecAssertions({
      @SpecAssertion(section="5.9", id = "b"),
      @SpecAssertion(section="5.9", id = "ga")
   })
   public void testDefaultBindingTypeAssumed() throws Exception
   {
      Set<Bean<Tuna>> possibleTargets = getCurrentManager().resolveByType(Tuna.class);
      assert possibleTargets.size() == 1;
      assert possibleTargets.iterator().next().getTypes().contains(Tuna.class);
   }
   
   @Test(groups="resolution", expectedExceptions=IllegalArgumentException.class) 
   @SpecAssertion(section="5.9", id = "cb")
   public void testParameterizedTypeWithWildcardParameter()
   {
      getCurrentManager().resolveByType(new TypeLiteral<ParameterizedBean_Broken<?>>(){});
   }
   
   @Test(groups="resolution", expectedExceptions=IllegalArgumentException.class) 
   @SpecAssertion(section="5.9", id = "ca")
   public  <T> void testParameterizedTypeWithTypeParameter()
   {
      getCurrentManager().resolveByType(new TypeLiteral<ParameterizedBean_Broken<T>>(){});
   }
   
   @Test(groups="resolution", expectedExceptions=DuplicateBindingTypeException.class) 
   @SpecAssertion(section="5.9", id = "d")
   public void testDuplicateBindingTypesUsed()
   {
      getCurrentManager().resolveByType(Tuna.class, new CurrentBinding(), new CurrentBinding());
   }
   
   @Test(groups="resolution", expectedExceptions=IllegalArgumentException.class) 
   @SpecAssertion(section="5.9", id = "e")
   public void testNonBindingTypeUsed()
   {
      getCurrentManager().resolveByType(Tuna.class, new AnotherDeploymentTypeLiteral());
   }
   
   @Test(groups="resolution") 
   @SpecAssertions({
      @SpecAssertion(section="5.9", id = "a"),
      @SpecAssertion(section = "5.9", id ="fa")
   })
   public void testResolveByType() throws Exception
   {
      
      assert getCurrentManager().resolveByType(Tuna.class, new CurrentBinding()).size() == 1;
      
      assert getCurrentManager().resolveByType(Tuna.class).size() == 1;
      
      Set<Bean<Animal>> beans = getCurrentManager().resolveByType(Animal.class, new AnnotationLiteral<FishILike>() {});
      assert beans.size() == 3;
      List<Class<? extends Animal>> classes = new ArrayList<Class<? extends Animal>>();
      for (Bean<Animal> bean : beans)
      {
         if (bean.getTypes().contains(Salmon.class))
         {
            classes.add(Salmon.class);
         }
         else if (bean.getTypes().contains(SeaBass.class))
         {
            classes.add(SeaBass.class);
         }
         else if (bean.getTypes().contains(Haddock.class))
         {
            classes.add(Haddock.class);
         }
      }
      assert classes.contains(Salmon.class);
      assert classes.contains(SeaBass.class);
      assert classes.contains(Haddock.class);
   }
   
   @Test(groups="injection") 
   @SpecAssertions({
     @SpecAssertion(section = "2.3.5", id = "b" ),
     @SpecAssertion(section = "5.9",   id = "gb"),
     @SpecAssertion(section = "2.3.3", id = "d" ),
     @SpecAssertion(section = "5.9.2", id = "a" )
   })
   public void testAllBindingTypesSpecifiedForResolutionMustAppearOnWebBean()
   {
      assert getCurrentManager().resolveByType(Animal.class, new ChunkyLiteral() {

         public boolean realChunky()
         {
            return true;
         }
         
      }, new AnnotationLiteral<Whitefish>() {}).size() == 1;
      assert getCurrentManager().resolveByType(Animal.class, new ChunkyLiteral() {

         public boolean realChunky()
         {
            return true;
         }
         
      }, new AnnotationLiteral<Whitefish>() {}).iterator().next().getTypes().contains(Cod.class);
      
      assert getCurrentManager().resolveByType(ScottishFish.class, new AnnotationLiteral<Whitefish>() {}).size() == 2;
      List<Class<? extends Animal>> classes = new ArrayList<Class<? extends Animal>>();
      for (Bean<ScottishFish> bean : getCurrentManager().resolveByType(ScottishFish.class, new AnnotationLiteral<Whitefish>() {}))
      {
         if (bean.getTypes().contains(Cod.class))
         {
            classes.add(Cod.class);
         }
         else if (bean.getTypes().contains(Sole.class))
         {
            classes.add(Sole.class);
         }
      }
      assert classes.contains(Cod.class);
      assert classes.contains(Sole.class);
   }
   
   @Test(groups="resolution")
   @SpecAssertions({
      @SpecAssertion(section="5.9", id = "fd"),
      @SpecAssertion(section="2.2", id="bb"),
      @SpecAssertion(section="2.2", id="ba")
   })
   public void testResolveByTypeWithTypeParameter() throws Exception
   {
      assert getCurrentManager().resolveByType(new TypeLiteral<Farmer<ScottishFish>>(){}).size() == 1;
      assert getCurrentManager().resolveByType(new TypeLiteral<Farmer<ScottishFish>>(){}).iterator().next().getTypes().contains(ScottishFishFarmer.class);
   }
   
   @Test(groups={"resolution", "producerMethod"}) 
   @SpecAssertion(section="5.9", id = "fc")
   public void testResolveByTypeWithArray() throws Exception
   {
      assert getCurrentManager().resolveByType(Spider[].class).size() == 1;
   }
   
   @Test @SpecAssertion(section="5.9", id = "ha")
   public void testOnlyHighestEnabledPrecedenceWebBeansResolved() throws Exception
   {
      assert getCurrentManager().resolveByType(Animal.class, new AnnotationLiteral<Whitefish>() {}).size() == 1;
      assert getCurrentManager().resolveByType(Animal.class, new AnnotationLiteral<Whitefish>() {}).iterator().next().getTypes().contains(Plaice.class);
      
   }
  
   
   @Test(groups="resolution") 
   @SpecAssertion(section="5.9", id = "hb")
   public void testNoWebBeansFound() throws Exception
   {
      assert getCurrentManager().resolveByType(Tuna.class, new AnnotationLiteral<FakeFish>() {}).size() == 0;
   }
   
   @Test(groups="resolution") 
   @SpecAssertions({
      @SpecAssertion(section="5.9.1", id = "b"),
      @SpecAssertion(section="5.9", id = "gc"),
      @SpecAssertion(section="5.9", id = "gd")
   })
   public void testResolveByTypeWithNonBindingMembers() throws Exception
   {
      
      Set<Bean<Animal>> beans = getCurrentManager().resolveByType(Animal.class, new ExpensiveLiteral() 
      {

         public int cost()
         {
            return 60;
         }

         public boolean veryExpensive()
         {
            return true;
         }
         
      }, new AnnotationLiteral<Whitefish>() {});
      assert beans.size() == 2;
      
      List<Class<? extends Animal>> classes = new ArrayList<Class<? extends Animal>>();
      for (Bean<Animal> bean : beans)
      {
         if (bean.getTypes().contains(Halibut.class))
         {
            classes.add(Halibut.class);
         }
         else if (bean.getTypes().contains(RoundWhitefish.class))
         {
            classes.add(RoundWhitefish.class);
         }
      }
      assert classes.contains(Halibut.class);
      assert classes.contains(RoundWhitefish.class);
   }
   
   @Test(groups="resolution", expectedExceptions=DefinitionException.class) 
   @SpecAssertion(section="5.9.1", id = "ca")
   public void testArrayValuedAnnotationMemberWithoutNonBinding()
   {
      getCurrentManager().resolveByType(Animal.class, new BindingTypeWithBindingArrayTypeMemberLiteral_Broken() {
         
         public boolean[] bool()
         {
            return new boolean[0];
         }
         
      });
   }
   
   @Test(groups="resolution", expectedExceptions=DefinitionException.class) 
   @SpecAssertion(section="5.9.1", id = "cb")
   public void testAnnotationValuedAnnotationMemberWithoutNonBinding()
   {
      getCurrentManager().resolveByType(Animal.class, new BindingTypeWithBindingAnnotationMemberLiteral_Broken()
      {
         
         public Expensive expensive()
         {
            return new ExpensiveLiteral()
            {
               public int cost()
               {
                  return 0;
               }
               
               public boolean veryExpensive()
               {
                  return false;
               }
            };
         }
      
      });
   }
      
}
