package org.jboss.jsr299.tck.tests.event;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Set;

import javax.context.Dependent;
import javax.event.Event;
import javax.event.Observer;
import javax.inject.DuplicateBindingTypeException;
import javax.inject.Standard;
import javax.inject.manager.Bean;
import javax.inject.manager.Manager;

import org.hibernate.tck.annotations.SpecAssertion;
import org.hibernate.tck.annotations.SpecAssertions;
import org.jboss.jsr299.tck.AbstractJSR299Test;
import org.jboss.testharness.impl.packaging.Artifact;
import org.jboss.testharness.impl.packaging.jsr299.BeansXml;
import org.testng.annotations.Test;

/**
 * Event bus tests
 * 
 * @author Nicklas Karlsson
 * @author David Allen
 * 
 *         Spec version: Public Release Draft 2
 */
@Artifact
@BeansXml("beans.xml")
public class EventTest extends AbstractJSR299Test
{

   public static class AnEventType
   {
   }

   public static class ATemplatedEventType<T>
   {
   }

   public static class AnObserver implements Observer<AnEventType>
   {
      public boolean wasNotified = false;

      public void notify(AnEventType event)
      {
         wasNotified = true;
      }
   }

   public static class AListObserver implements Observer<ArrayList<String>>
   {
      public boolean wasNotified = false;

      public void notify(ArrayList<String> event)
      {
         wasNotified = true;
      }
   }

   public static class AnotherListObserver implements Observer<ArrayList<Integer>>
   {
      public boolean wasNotified = false;

      public void notify(ArrayList<Integer> event)
      {
         wasNotified = true;
      }
   }

   public static class AnObserverWithException implements Observer<AnEventType>
   {
      public boolean wasNotified = false;
      public RuntimeException theException = new RuntimeException("RE1");

      public void notify(AnEventType event)
      {
         wasNotified = true;
         throw theException;
      }
   }

   /**
    * An event binding type is a Java annotation defined as @Target({FIELD,
    * PARAMETER}) ~or @Target({METHOD, FIELD, PARAMETER, TYPE}) and
    * 
    * @Retention(RUNTIME)~
    */
   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.1", id = "d") })
   public void testEventBindingTypeTargetMostAndRuntime()
   {

   }

   /**
    * An event binding type is a Java annotation defined as ~@Target({FIELD,
    * PARAMETER}) or~ @Target({METHOD, FIELD, PARAMETER, TYPE}) ~and
    * 
    * @Retention(RUNTIME)~
    */
   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.1", id = "e") })
   public void testEventBindingTypeTargetsFieldParameterAndRuntime()
   {

   }

   /**
    * An event binding type is a Java annotation defined as ~@Target({FIELD,
    * PARAMETER}) or @Target({METHOD, FIELD, PARAMETER, TYPE}) and
    * ~@Retention(RUNTIME) TODO If the binding does not have RUNTIME retention,
    * it cannot be tested
    */
   @Test(groups = { "events", "broken" })
   @SpecAssertions( { @SpecAssertion(section = "7.1", id = "f") })
   public void testEventBindingTypeNotRuntime()
   {

   }

   // TODO How to test all annotations w/o BindingType are not event bindings
   @Test(groups = { "events", "broken" })
   @SpecAssertions( { @SpecAssertion(section = "7.1", id = "g") })
   public void testEventBindingTypesSpecifyBinding()
   {

   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertions( { @SpecAssertion(section = "7.1", id = "a"), @SpecAssertion(section = "7.2", id = "b") })
   public void testManagerFireEventWithEventTypeParametersFails()
   {
      ATemplatedEventType<String> anEvent = new ATemplatedEventType<String>();
      getCurrentManager().fireEvent(anEvent);
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertions( { @SpecAssertion(section = "7.1", id = "b"), @SpecAssertion(section = "7.2", id = "c") })
   public void testManagerFireEventWithEventTypeWildcardsFails()
   {
      // Although the above test is really the same as with a wildcard,
      // we will test it anyhow since the specification calls it out separately.
      ATemplatedEventType<?> anEventOnAnyType = new ATemplatedEventType<String>();
      getCurrentManager().fireEvent(anEventOnAnyType);
   }

   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.2", id = "a") })
   public void testManagerInterfaceForFireEventMethod() throws Exception
   {
      assert Manager.class.getDeclaredMethod("fireEvent", Object.class, Annotation[].class) != null;
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertions( { @SpecAssertion(section = "7.2", id = "d") })
   public void testManagerFireEventWithNonBindingAnnotationsFails()
   {
      // The specs are not exactly clear on what is supposed to happen here,
      // but borrowing from Section 8.3, we'll expect the same behavior here
      // for a consistent API.
      // TODO Verify that fireEvent should fail on non-binding annotations
      AnEventType anEvent = new AnEventType();
      getCurrentManager().fireEvent(anEvent, new AnimalStereotypeAnnotationLiteral());
   }

   /**
    * The Manager interface provides a method for firing events:
    */
   @Test(groups = { "events" })
   @SpecAssertion(section = "7.2", id = "a")
   public void testManagerFireEvent()
   {
      // First a simple event with no bindings is fired
      AnEventType anEvent = new AnEventType();
      getCurrentManager().fireEvent(anEvent);

      // Next an event with some event bindings is fired
      getCurrentManager().fireEvent(anEvent, new RoleBinding("Admin"));
   }

   // TODO This is not an assertion but a definition which is circular
   @Test(groups = { "events" })
   @SpecAssertion(section = "7.3", id = "a")
   public void testObserversImplementObserverInterface()
   {
      Observer<String> observer = new Observer<String>()
      {
         public void notify(String event)
         {
         }
      };
      assert observer != null;
   }

   @Test(groups = { "events", "broken" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertions( { @SpecAssertion(section = "7.3", id = "f") })
   public void testManagerAddObserverWithEventTypeParametersFails()
   {
      assert false;
      // ATemplatedEventType<String> anEvent = new
      // ATemplatedEventType<String>();
      // Observer<AnEventType> observer = new AnObserver();
      // getCurrentManager().addObserver(observer, anEvent.getClass());
   }

   @Test(groups = { "events", "broken" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertions( { @SpecAssertion(section = "7.3", id = "g") })
   public void testManagerAddObserverWithEventTypeWildcardsFails()
   {
      // Although the above test is really the same as with a wildcard,
      // we will test it anyhow since the specification calls it out separately.
      ATemplatedEventType<?> anEventOnAnyType = new ATemplatedEventType<String>();
      assert false;
   }

   @Test(groups = { "events", "broken" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertions( { @SpecAssertion(section = "7.3", id = "h") })
   public void testManagerRemoveObserverWithEventTypeParametersFails()
   {
      assert false;
      // ATemplatedEventType<String> anEvent = new
      // ATemplatedEventType<String>();
      // Observer<AnEventType> observer = new AnObserver();
      // getCurrentManager().addObserver(observer, anEvent.getClass());
   }

   @Test(groups = { "events", "stub" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertions( { @SpecAssertion(section = "7.3", id = "i") })
   public void testManagerRemoveObserverWithEventTypeWildcardsFails()
   {
      // Although the above test is really the same as with a wildcard,
      // we will test it anyhow since the specification calls it out separately.
      ATemplatedEventType<?> anEventOnAnyType = new ATemplatedEventType<String>();
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.7", id = "a") })
   public void testManagerResolveObserversSignature() throws Exception
   {
      assert getCurrentManager().getClass().getDeclaredMethod("resolveObservers", Object.class, Annotation[].class) != null;
   }

   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.4", id = "a") })
   public void testObserverNotificationCallsResolveObservers()
   {
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.5", id = "a"), @SpecAssertion(section = "7.5.2", id = "a")} )
   public void testObserverMethodAutomaticallyRegistered()
   {
      assert !getCurrentManager().resolveObservers("event").isEmpty();
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5", id = "e")
   public void testMultipleObserverMethodsForSameEventOK()
   {
      assert getCurrentManager().resolveObservers("event").size() > 1;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5", id = "f")
   public void testMultipleObserverMethodsOnBeanOK()
   {
      // Resolve the observers for types 1 and 2
      Set<Observer<ObservedType1>> resolvedStringObservers = getCurrentManager().resolveObservers(new ObservedType1());
      assert resolvedStringObservers.size() == 1;

      Set<Observer<ObservedType2>> resolvedIntegerObservers = getCurrentManager().resolveObservers(new ObservedType2());
      assert resolvedIntegerObservers.size() == 1;
   }

   @Test(groups = { "events", "ejb" })
   @SpecAssertion(section = "7.5", id = "d")
   public void testObserverMethodOnEnterpriseBeanIsBusinessMethodOrStatic()
   {
      Set<Observer<EJBEvent>> observers = getCurrentManager().resolveObservers(new EJBEvent());
      assert observers.size() == 2;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5.1", id = "b")
   public void testObserverMethodWithoutBindingTypesObservesEventsWithoutBindingTypes()
   {
      // Resolve registered observers with an event containing no binding types
      Set<Observer<SimpleEventType>> resolvedObservers = getCurrentManager().resolveObservers(new SimpleEventType());
      assert resolvedObservers.size() == 2;
   }

   @Test(groups = { "events" })
   @SpecAssertions( { 
      @SpecAssertion(section = "7.5.2", id = "g"), 
      @SpecAssertion(section = "7.7.1", id = "a"), 
      @SpecAssertion(section = "7.7.2", id = "a") 
   })
   public void testObserverMethodMayHaveMultipleBindingTypes()
   {
      // If we can resolve the observer with the two binding types,
      // then it worked
      Set<Observer<MultiBindingType>> resolvedObservers = getCurrentManager().resolveObservers(new MultiBindingType(), new RoleBinding("Admin"), new TameAnnotationLiteral());
      assert resolvedObservers.size() == 2;
   }

//   @Test(groups = { "events", "broken" })
//   @SpecAssertions( { @SpecAssertion(section = "7.7.1", id = "b") } )
//   public void testEqualsUsedToCompareBindingTypes()
//   {
//      RoleBinding roleBinding = new RoleBinding("Admin");
//      getCurrentManager().resolveObservers(new MultiBindingType(), roleBinding, new TameAnnotationLiteral());
//      assert false;
//   }

   @Test(groups = { "stub", "events", "webbeansxml" })
   @SpecAssertions( { 
      @SpecAssertion(section = "7.5.3", id = "a"), 
      @SpecAssertion(section = "7.5.3", id = "b") 
   } )
   public void testXMLDefinedObserverMethodIgnoresBindingAnnotations()
   {
      assert false;
   }

   @Test(groups = { "stub", "events", "webbeansxml" })
   @SpecAssertion(section = "7.5.3", id = "c")
   public void testXMLDefinedObserverNotFindingImplementationMethodFails()
   {
      assert false;
   }

   /**
    * In addition to the event parameter, observer methods may declare
    * additional parameters, which may declare bindings. The container calls the
    * method Manager.getInstanceToInject() defined in Section 5.7.1, "Resolving
    * dependencies" to determine a value for each parameter of an observer
    * method and calls the observer method with those parameter values.
    * 
    * To invoke an observer method, the container must pass the event object to
    * the event parameter and the object returned by
    * Manager.getInstanceToInject() to each of the other parameters.
    */
   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.4", id = "a"), @SpecAssertion(section = "7.5.8", id = "j") })
   public void testObserverMethodReceivesInjectionsOnNonObservesParameters()
   {
      Set<Bean<Object>> beans = getCurrentManager().resolveByType(Object.class);
      assert beans != null;
   }

   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.5", id = "a"), @SpecAssertion(section = "7.5.5", id = "b")} )
   public void testConditionalObserver() throws Exception
   {
      RecluseSpider.notified = false;
      getCurrentManager().fireEvent(new ConditionalEvent());
      // Should not be notified since bean is not instantiated yet
      assert !RecluseSpider.notified;

      // Now instantiate the bean and fire another event
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            RecluseSpider bean = getCurrentManager().getInstanceByType(RecluseSpider.class);
            assert bean != null;
            // Must invoke a method to really create the instance
            assert !bean.isNotified();

            getCurrentManager().fireEvent(new ConditionalEvent());
            assert bean.isNotified();
         }

      }.run();
   }

   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.5", id = "c") } )
   public void testConditionalObserverDefinedByXML()
   {
      assert false;
   }
   
   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.7", id = "a"), @SpecAssertion(section = "7.5.8", id = "c"), @SpecAssertion(section = "7.5.8", id = "g")} )
   public void testAsynchronousObserverIsAsynchronous() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            getCurrentManager().fireEvent(new Boolean(true));
            Thread.sleep(200);
            assert !AsynchronousObserver.getThreadObservingEvent().equals(Thread.currentThread());
         }
         
      }.run();
   }

   @Test(groups = { "stub", "events", "webbeansxml" })
   @SpecAssertion(section = "7.5.7", id = "b")
   public void testAsynchronousObserverDeclaredByXML()
   {
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5.8", id = "a")
   public void testObserverMethodRegistration()
   {
      // Resolve registered observers with an event containing no binding types
      Set<Observer<SimpleEventType>> resolvedObservers = getCurrentManager().resolveObservers(new SimpleEventType());
      assert resolvedObservers.size() == 2;
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.8", id = "b"), @SpecAssertion(section = "7.5.8", id = "h")} )
   public void testObserverMethodCalledImmediately()
   {

      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.5", id = "c"), @SpecAssertion(section = "7.5.8", id = "k")} )
   public void testStaticObserverMethodInvoked()
   {
      getCurrentManager().fireEvent(new Delivery());
      assert StaticObserver.isDeliveryReceived();
   }

   /**
    * Otherwise, if the observer method is non-static, the container must:
    * 
    * • obtain the Bean object for the most specialized bean that specializes
    * the bean which declares the observer method, and then
    * 
    */
   @Test(groups = { "events" })
   @SpecAssertions({
      @SpecAssertion(section = "7.5.8", id = "l"),
      @SpecAssertion(section = "4.3.2", id = "f")
   })
   public void testObserverCalledOnMostSpecializedInstance()
   {
      Shop.deliveryObservedBy = null;
      getCurrentManager().fireEvent(new Delivery());
      assert Shop.deliveryObservedBy.equals(FarmShop.class.getName());
   }

   /**
    * obtain the context object by calling Manager.getContext(), passing the
    * bean scope, then
    */
   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.8", id = "m")
   public void testObserverContextRetrieved()
   {
      assert false;
   }

   /**
    * obtain an instance of the bean by calling Context.get(), passing the Bean
    * instance representing the bean, together with a CreationalContext unless
    * this observer method is a conditional observer method, and then
    */
   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.8", id = "n")
   public void testObserverInstanceRetrievedFromContext()
   {
      assert false;
   }

   /**
    * if the get() method returned a non-null value, invoke the observer method
    * on the returned instance
    */
   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.8", id = "o")
   public void testObserverMethodInvokedOnReturnedInstanceFromContext()
   {
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.5.8", id = "p")
   public void testAsynchronousObserverThrowsExceptionIsLogged() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            getCurrentManager().fireEvent('a');
            Thread.sleep(200);
            //TODO How can we assert that the exception was logged?
         }
         
      }.run();
   }

   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.9", id = "a")
   public void testAsynchronousObserverMethodContexts()
   {
      assert false;
   }

   @Test(groups = { "events" }, expectedExceptions = { DuplicateBindingTypeException.class })
   @SpecAssertion(section = "7.6", id = "g")
   public void testDuplicateBindingsToFireFails() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            SweeWaxbill bean = getCurrentManager().getInstanceByType(SweeWaxbill.class);
            bean.methodThatFiresEvent();
         }

      }.run();
   }

   @Test(groups = { "events" }, expectedExceptions = { DuplicateBindingTypeException.class })
   @SpecAssertion(section = "7.6", id = "g")
   public void testDuplicateBindingsToObservesFails() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            SweeWaxbill bean = getCurrentManager().getInstanceByType(SweeWaxbill.class);
            bean.methodThatRegistersObserver();
         }

      }.run();
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.6", id = "h")
   public void testNonBindingTypePassedToFireFails() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            OwlFinch_Broken bean = getCurrentManager().getInstanceByType(OwlFinch_Broken.class);
            bean.methodThatFiresEvent();
         }

      }.run();
   }

   @Test(groups = { "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.6", id = "h")
   public void testNonBindingTypePassedToObservesFails() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            OwlFinch_Broken bean = getCurrentManager().getInstanceByType(OwlFinch_Broken.class);
            bean.methodThatRegistersObserver();
         }

      }.run();
   }

   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.6", id = "n") } )
   public void testImplicitEventBeanMatchesAPITypeOfInectionPoint() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            // Retrieve the implicit event bean from the manager only by
            // its API type
            Set<?> eventBeans = getCurrentManager().resolveByType(Event.class, new FiresBinding());
            assert !eventBeans.isEmpty();
         }

      }.run();
   }

   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.6", id = "n") } )
   public void testImplicitEventBeanMatchesBindingAnnotationsOfInjectionPoint() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            // Retrieve the implicit event bean from the manager
            // by its binding types (uses OrangeCheekedWaxbill)
            Set<?> eventBeans = getCurrentManager().resolveByType(Event.class, new FiresBinding(), new TameAnnotationLiteral(), new RoleBinding("Admin"));
            assert !eventBeans.isEmpty();
         }

      }.run();
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.6", id = "o")
   public void testImplicitEventBeanHasStandardDeploymentType() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            // Retrieve the implicit event bean from the manager
            // only by its API type (uses BlueFacedParrotFinch)
            Set<?> eventBeans = getCurrentManager().resolveByType(Event.class, new FiresBinding());
            assert eventBeans.size() == 1;
            Bean<?> eventBean = (Bean<?>) eventBeans.iterator().next();
            assert eventBean.getDeploymentType().equals(Standard.class);
         }

      }.run();
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.6", id = "p")
   public void testImplicitEventBeanHasDependentScope() throws Exception
   {
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            // Retrieve the implicit event bean from the manager only
            // by its API type (uses BlueFacedParrotFinch)
            Set<?> eventBeans = getCurrentManager().resolveByType(Event.class, new FiresBinding());
            assert eventBeans.size() == 1;
            Bean<?> eventBean = (Bean<?>) eventBeans.iterator().next();
            assert eventBean.getScopeType().equals(Dependent.class);
         }
      }.run();
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.6", id = "q")
   public void testImplicitEventBeanHasNoName() throws Exception
   {
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            // Retrieve the implicit event bean from the manager only
            // by its API type (uses BlueFacedParrotFinch)
            Set<?> eventBeans = getCurrentManager().resolveByType(Event.class, new FiresBinding());
            assert eventBeans.size() == 1;
            Bean<?> eventBean = (Bean<?>) eventBeans.iterator().next();
            assert eventBean.getName() == null;
         }
      }.run();
   }

   @Test(groups = { "events" })
   @SpecAssertions( { @SpecAssertion(section = "7.6", id = "a"), @SpecAssertion(section = "7.6", id = "r") } )
   public void testImplicitEventBeanProvidedByContainer() throws Exception
   {
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            // Retrieve the implicit event bean from the manager only
            // by its API type (uses BlueFacedParrotFinch)
            Set<?> eventBeans = getCurrentManager().resolveByType(Event.class, new FiresBinding());
            assert eventBeans.size() == 1;
         }
      }.run();
   }

   // TODO The manager related tests require intercepting all calls
   // to it to verify that the correct calls are made.
   @Test(groups = { "events", "underInvestigation" })
   @SpecAssertion(section = "7.6", id = "s")
   public void testFireMethodCallsManagerFireWithEventObject()
   {
      assert false;
   }

   @Test(groups = { "events", "underInvestigation" })
   @SpecAssertion(section = "7.6", id = "s")
   public void testFireMethodCallsManagerFireWithBindingAnnotationsExceptFires()
   {
      assert false;
   }

   @Test(groups = { "events", "underInvestigation" })
   @SpecAssertion(section = "7.6", id = "s")
   public void testFireMethodCallsManagerFireWithAllBindingAnnotationInstances()
   {
      assert false;
   }

   @Test(groups = { "events", "underInvestigation" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.8", id = "s"), @SpecAssertion(section = "7.6", id = "u")} )
   public void testObserverMethodCallsManagerAddObserverWithObserverObject()
   {
      assert false;
   }

   @Test(groups = { "events", "underInvestigation" })
   @SpecAssertion(section = "7.6", id = "u")
   public void testObserverMethodCallsManagerAddObserverWithAllBindingAnnotationsExceptFires()
   {
      assert false;
   }

   @Test(groups = { "events", "underInvestigation" })
   @SpecAssertion(section = "7.6", id = "u")
   public void testObserverMethodCallsManagerAddObserverWithAllBindingAnnotationInstance()
   {
      assert false;
   }

   @Test(groups = { "events" })
   @SpecAssertion(section = "7.7", id = "b")
   public void testEventObjectContainsTypeVariablesWhenResolvingFails()
   {
      eventObjectContainsTypeVariables(new ArrayList<String>());
   }

   private <E> void eventObjectContainsTypeVariables(ArrayList<E> eventToFire)
   {
      getCurrentManager().resolveObservers(eventToFire);
   }

   @Test(groups = { "broken", "events" }, expectedExceptions = { IllegalArgumentException.class })
   @SpecAssertion(section = "7.7", id = "c")
   public void testEventObjectContainsWildcardsWhenResolvingFails()
   {
      eventObjectContainsWildcards(new ArrayList<String>());
   }

   private void eventObjectContainsWildcards(ArrayList<?> eventToFire)
   {
      // TODO There does not seem to be a way to get wildcarded types passed
      getCurrentManager().resolveObservers(eventToFire);
   }

   
   @Test(groups = { "events" })
   @SpecAssertions( { 
      @SpecAssertion(section = "7.7.2", id = "b"), 
      @SpecAssertion(section = "7.7.2", id = "c")
   } )
   public void testObserverMethodNotifiedWhenBindingsMatch()
   {
      getCurrentManager().fireEvent(new MultiBindingType(), new RoleBinding("Admin"), new TameAnnotationLiteral());
      assert BullTerrier.isMultiBindingEventObserved();
      assert BullTerrier.isSingleBindingEventObserved();
   }

   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.8", id = "a") })
   public void testEventTypeMappedToJmsTopic()
   {
      assert false;
   }

   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.8", id = "b") })
   public void testMultipleEventTypesMappedToSameJmsTopic()
   {
      assert false;
   }

   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.8", id = "d") })
   public void testSerializedEventAndBindingsSentToJmsTopic()
   {
      assert false;
   }

   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.8", id = "e") })
   public void testObserversNotifiedOfEventFromJmsTopic()
   {
      assert false;
   }

   @Test(groups = { "events", "stub" })
   @SpecAssertions( { @SpecAssertion(section = "7.8", id = "f") })
   public void testEventsDistributedToProcessesWithSameTypeAndJmsTopic()
   {
      assert false;
   }

   /**
    * By default, Java implementation reuse is assumed. In this case, the
    * producer, disposal and observer methods of the first bean are not
    * inherited by the second bean.
    * 
    * @throws Exception
    */
   @Test(groups = { "events", "inheritance" })
   @SpecAssertion(section = "4.2", id = "dc")
   public void testNonStaticObserverMethodNotInherited() throws Exception
   {
      new RunInDependentContext()
      {

         @Override
         protected void execute() throws Exception
         {
            Set<Observer<Egg>> observers = getCurrentManager().resolveObservers(new Egg());
            assert observers.size() == 1;

            // Notify the observer so we can confirm that it
            // is a method only on Farmer, and not LazyFarmer
            observers.iterator().next().notify(new Egg());
            assert Farmer.getObserverClazz().equals(Farmer.class);
         }

      }.run();
   }

}
