/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,  
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.jsr299.tck.tests.event.transactionalObservers;

import java.lang.annotation.Annotation;
import java.math.BigInteger;

import javax.inject.AnnotationLiteral;

import org.hibernate.tck.annotations.SpecAssertion;
import org.hibernate.tck.annotations.SpecAssertions;
import org.jboss.jsr299.tck.AbstractJSR299Test;
import org.jboss.testharness.impl.packaging.Artifact;
import org.jboss.testharness.impl.packaging.IntegrationTest;
import org.jboss.testharness.impl.packaging.Packaging;
import org.jboss.testharness.impl.packaging.PackagingType;
import org.testng.annotations.Test;

/**
 * Integration tests for Web Bean events.
 * 
 * @author David Allen
 * 
 *         Spec version: Public Release Draft 2
 * 
 */
@Artifact
@Packaging(PackagingType.EAR)
@IntegrationTest
@Test
public class EventTest extends AbstractJSR299Test
{

   private static final Annotation TAME_LITERAL = new AnnotationLiteral<Tame>()
   {
   };
   private PomeranianInterface dog = null;
   
   public void beforeMethod() 
   {
      super.beforeMethod();
      dog = (PomeranianInterface) getCurrentManager().getInstanceByName("Teddy");
   }

//   @AfterMethod(alwaysRun = true)
//   public void teardownTest()
//   {
//      try
//      {
//         dog.removeSessionBean();
//      }
//      catch (Exception e)
//      {
//         // Not important since the bean is now gone one way or the other
//      }
//   }

   @Test(groups = { "events", "integration" })
   @SpecAssertion(section = "7.5.6", id = "a")
   public void testTransactionalObserverNotifiedImmediatelyWhenNoTransactionInProgress()
   {
      dog.setCorrectContext(false);
      dog.setCorrectTransactionState(false);
      Agent dogAgent = getCurrentManager().getInstanceByType(Agent.class);
      assert dogAgent != null;
      dogAgent.sendOutsideTransaction(BigInteger.TEN);
      assert dog.isCorrectTransactionState();
      // TODO Fix the security contexts
      // assert dog.isCorrectContext();
   }

   @Test(groups = { "events", "integration" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.6", id = "c"), @SpecAssertion(section = "7.5.6", id = "f"), @SpecAssertion(section = "7.5.6", id = "i"), @SpecAssertion(section = "7.5.8", id = "d"), @SpecAssertion(section = "7.5.8", id = "f") })
   public void testAfterTransactionCompletionObserver() throws InterruptedException
   {
      dog.setCorrectContext(false);
      dog.setCorrectTransactionState(false);
      assert !getCurrentManager().resolveObservers("event").isEmpty();
      Agent dogAgent = getCurrentManager().getInstanceByType(Agent.class);
      dogAgent.sendInTransaction("event");
      Thread.sleep(100);
      assert dog.isCorrectTransactionState();
   }

   @Test(groups = { "events", "integration", "broken" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.6", id = "d"), @SpecAssertion(section = "7.5.6", id = "j") })
   public void testAfterTransactionSuccessObserver() throws InterruptedException
   {
      dog.setCorrectContext(false);
      dog.setCorrectTransactionState(false);
      Agent dogAgent = getCurrentManager().getInstanceByType(Agent.class);
      dogAgent.sendInTransaction(new Integer(4));
      Thread.sleep(100);
      assert dog.isCorrectTransactionState();
   }

   @Test(groups = { "events", "integration", "ri-broken" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.6", id = "e"), @SpecAssertion(section = "7.5.6", id = "k") })
   public void testAfterTransactionFailureObserver() throws InterruptedException
   {
      dog.setCorrectContext(false);
      dog.setCorrectTransactionState(false);
      Agent dogAgent = getCurrentManager().getInstanceByType(Agent.class);
      dogAgent.sendInTransaction(new Float(4.0));
      Thread.sleep(100);
      assert dog.isCorrectTransactionState();
   }

   @Test(groups = { "events", "integration" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.6", id = "b"), @SpecAssertion(section = "7.5.6", id = "h"), @SpecAssertion(section = "7.5.8", id = "e") })
   public void testBeforeTransactionCompletionObserver()
   {
      dog.setCorrectContext(false);
      dog.setCorrectTransactionState(false);
      Agent dogAgent = getCurrentManager().getInstanceByType(Agent.class);
      dogAgent.sendInTransaction(new RuntimeException("test event"));
      assert dog.isCorrectTransactionState();
      // TODO Fix the security contexts
      // assert dog.isCorrectContext();
   }

   @Test(groups = { "stub", "events", "integration" })
   @SpecAssertions( { @SpecAssertion(section = "7.5.6", id = "g") })
   public void testTransactionalObserverDefinedByXML()
   {
      assert false;
   }

   @Test(groups = { "events", "broken" })
   @SpecAssertion(section = "7.5.7", id = "c")
   public void testAsynchronousObserverAlsoTransactional() throws InterruptedException
   {
      dog.setCorrectContext(false);
      dog.setCorrectTransactionState(false);
      Agent dogAgent = getCurrentManager().getInstanceByType(Agent.class);
      dogAgent.sendInTransaction('a');
      Thread.sleep(200);
      assert dog.isCorrectTransactionState();
   }

   /**
    * If the observer method is an asynchronous transactional observer method
    * and there is currently a JTA transaction in progress, the observer object
    * calls the observer method asynchronously during the after transaction
    * completion phase.
    */
   @Test(groups = { "stub", "events" })
   @SpecAssertion(section = "7.5.8", id = "e")
   public void testAsynchronousTransactionalObserverCalledAfterCompletion()
   {
      assert false;
   }

   @Test(groups = { "broken", "events", "integration" })
   @SpecAssertion(section = "7.5.8", id = "p")
   public void testTransactionalObserverThrownExceptionIsCaughtAndLogged()
   {
      // TODO There really is no way to verify that something is logged
      assert false;
   }

   /**
    * Otherwise, the observer method is called in the same transaction context,
    * client security context and lifecycle contexts as the invocation of
    * Event.fire().
    */
   @Test(groups = { "stub", "events", "integration" })
   @SpecAssertion(section = "7.5.9", id = "d")
   public void testObserverMethodCalledInSameContexts()
   {
      assert false;
   }

   /**
    * Otherwise, if the observer method is a @BeforeTransactionCompletion
    * transactional observer method, it is called within the context of the
    * transaction that is about to complete and with the same client security
    * context and lifecycle contexts.
    */
   @Test(groups = { "stub", "events", "integration" })
   @SpecAssertion(section = "7.5.9", id = "b")
   public void testBeforeTransactionCompletionObserverMethodContexts()
   {
      assert false;
   }

   /**
    * Otherwise, if the observer method is any other kind of transactional
    * observer method, it is called in an unspecified transaction context, but
    * with the same client security context and lifecycle contexts as the
    * transaction that just completed.
    */
   @Test(groups = { "stub", "events", "integration" })
   @SpecAssertion(section = "7.5.9", id = "c")
   public void testOtherTransactionalObserverMethodContexts()
   {
      assert false;
   }
   
   @Test(groups = { "stub", "events", "integration" })
   @SpecAssertion(section = "7.4", id = "d")
   public void testObserverCanSetRollbackOnlyOnTransaction()
   {
      assert false;
   }

   @Test(groups = { "stub", "events", "integration" })
   @SpecAssertion(section = "7.4", id = "e")
   public void testObserverCannotInitiateJtaTransaction()
   {
      assert false;
   }

   @Test(groups = { "stub", "events", "integration" })
   @SpecAssertion(section = "7.4", id = "f")
   public void testObserverCannotCommitJtaTransaction()
   {
      assert false;
   }

   @Test(groups = { "stub", "events", "integration" })
   @SpecAssertion(section = "7.4", id = "g")
   public void testObserverCannotRollbackJtaTransaction()
   {
      assert false;
   }
}
