/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.dependency.plugins;

import org.jboss.dependency.spi.Controller;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.dependency.spi.DependencyInfo;
import org.jboss.dependency.spi.DependencyItem;
import org.jboss.logging.Logger;
import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;

/**
 * A DependencyItem.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 * @version $Revision: 101410 $
 */
public class AbstractDependencyItem extends JBossObject implements DependencyItem
{
   /** The log */
   protected static final Logger log = Logger.getLogger(AbstractDependencyItem.class);
   
   /** What I depend on */
   private volatile Object iDependOn;

   /** My name */
   private volatile Object name;

   /** When the dependency is required */
   private volatile ControllerState whenRequired = ControllerState.DESCRIBED;

   /** The state of the dependency */
   private volatile ControllerState dependentState;
   
   /** Whether we optimized the states or not */
   private volatile boolean indexedStates; 
   
   /** Whether we are resolved */
   private boolean resolved;

   /**
    * Create a new dependency item
    */
   public AbstractDependencyItem()
   {
   }

   /**
    * Create a new dependency item
    *
    * @param name my name
    * @param iDependOn what I depend on
    * @param whenRequired when the dependency is required 
    * @param dependentState the required state of the dependent 
    */
   public AbstractDependencyItem(Object name, Object iDependOn, ControllerState whenRequired, ControllerState dependentState)
   {
      this.name = name;
      setIDependOn(iDependOn);
      this.whenRequired = whenRequired;
      this.dependentState = dependentState;
   }

   public Object getName()
   {
      return name;
   }

   public Object getIDependOn()
   {
      return iDependOn;
   }

   public ControllerState getWhenRequired()
   {
      return whenRequired;
   }

   public ControllerState getDependentState()
   {
      return dependentState;
   }

   public boolean isResolved()
   {
      return resolved;
   }
      
   public boolean resolve(Controller controller)
   {
      indexStates(controller, ControllerState.DESCRIBED, ControllerState.INSTALLED);
      boolean previous = resolved;
      ControllerState state = dependentState == null ? ControllerState.INSTALLED : dependentState;
      ControllerContext context = controller.getContext(iDependOn, state, true);
      
      if (context == null)
      {
         resolved = false;
      }
      else
      {
         addDependsOnMe(controller, context);
         resolved = true;
      }
      if (previous != resolved)
      {
         flushJBossObjectCache();
         if (log.isTraceEnabled())
         {
            if (resolved)
               log.trace("Resolved " + this);
            else
               log.trace("Unresolved " + this);
         }
      }
      return resolved;
   }

   public void unresolved()
   {
      if (resolved)
      {
         resolved = false;
         flushJBossObjectCache();
         log.trace("Forced unresolved " + this);
      }
   }
   
   public boolean unresolved(Controller controller)
   {
      unresolved();
      return true;
   }
   
   public void toString(JBossStringBuilder buffer)
   {
      buffer.append("name=").append(name);
      buffer.append(" dependsOn=").append(iDependOn);
      if (whenRequired != null)
         buffer.append(" whenRequired=").append(whenRequired.getStateString());
      if (dependentState != null)
         buffer.append(" dependentState=").append(dependentState.getStateString());
      buffer.append(" resolved=").append(resolved);
   }
   
   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(name).append(" dependsOn ").append(iDependOn);
   }

   /**
    * Register a dependency with another context
    * 
    * @param controller the controller
    * @param context the other context
    */
   protected void addDependsOnMe(Controller controller, ControllerContext context)
   {
      DependencyInfo info = context.getDependencyInfo();
      if (info != null)
         info.addDependsOnMe(this);
   }

   /**
    * Set what I depend upon 
    * 
    * @param iDependOn what I depend upon
    */
   protected void setIDependOn(Object iDependOn)
   {
      this.iDependOn = iDependOn;

      // only check if not null
      if (iDependOn != null)
      {
         // HACK: Try to fixup JMX like ObjectNames to their canonical name
         Object fixup = JMXObjectNameFix.needsAnAlias(iDependOn);
         if (fixup != null)
            this.iDependOn = fixup;
      }

      flushJBossObjectCache();
   }

   /**
    * Set the resolved state 
    * 
    * @param resolved the new resolved state
    */
   protected void setResolved(boolean resolved)
   {
      this.resolved = resolved;
      flushJBossObjectCache();
   }

   public String toHumanReadableString()
   {
      StringBuilder builder = new StringBuilder();
      toHumanReadableString(builder);
      return builder.toString();
   }

   /**
    * Add info to builder.
    *
    * @param builder the string builder
    */
   protected void toHumanReadableString(StringBuilder builder)
   {
      builder.append("Depends on '").append(getIDependOn()).append("'");
   }
   
   /**
    * Indexes the states for optimized lookups
    * 
    * @param controller the controller. If it is an instance of AbstractController the states will be indexed
    */
   protected void indexStates(Controller controller, ControllerState defaultWhenRequired, ControllerState defaultDependent)
   {
      if (indexedStates == false)
      {
         if (controller instanceof AbstractController == false)
         {
            log.warn("controller was not an AbstractController when indexing the states");
         }
         else
         {
            ControllerState req = whenRequired;
            if (req == null)
               req = defaultWhenRequired;
            ControllerState dep = dependentState;
            if (dep == null)
               dep = defaultDependent;

            AbstractController ac = (AbstractController) controller;
            whenRequired = ac.indexState(req);
            dependentState = ac.indexState(dep);
         }

         indexedStates = true;
      }
   }
}
