/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.dependency.plugins;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;

import org.jboss.dependency.spi.CallbackItem;
import org.jboss.dependency.spi.Controller;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.dependency.spi.DependencyItem;
import org.jboss.dependency.spi.LifecycleCallbackItem;
import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;
import org.jboss.util.collection.ConcurrentSet;

/**
 * A DependencyInfo.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 * @version $Revision: 101823 $
 */
public class AbstractDependencyInfo extends JBossObject implements TrackingDependencyInfo
{
   /** My dependencies */
   private volatile Set<DependencyItem> iDependOn;

   /** Dependencies referencing me */
   private volatile Set<DependencyItem> dependsOnMe;

   /** Install callbacks */
   private volatile Set<CallbackItem<?>> installCallbacks;

   /** Uninstall callbacks */
   private volatile Set<CallbackItem<?>> uninstallCallbacks;
   
   /** Lifecycle callbacks */
   private volatile List<LifecycleCallbackItem> lifecycleCallbacks;
   
   /** Whether this is an autowire candidate */
   private volatile boolean autowireCandidate = true;
   
   /** Whether we are tracking */
   volatile boolean tracking = true;
   
   /** The unresolved dependencies by state */
   private volatile Map<ControllerState, Set<DependencyItem>> unresolved;

   /**
    * Create an abstract dependency info
    */
   public AbstractDependencyInfo()
   {
   }

   public void resolved(DependencyItem item)
   {
      if (tracking)
      {
         ControllerState whenRequired = item.getWhenRequired();
         Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
         if (unresolved != null)
         {
            Set<DependencyItem> items = unresolved.get(whenRequired);
            if (items != null)
               items.remove(item);
         }
      }
   }

   public void unresolved(DependencyItem item)
   {
      if (tracking)
      {
         ControllerState whenRequired = item.getWhenRequired();
         Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
         if (unresolved == null)
         {
            this.unresolved = new ConcurrentHashMap<ControllerState, Set<DependencyItem>>();
            unresolved = this.unresolved;
         }
         Set<DependencyItem> items = unresolved.get(whenRequired);
         if (items == null)
         {
            items = new ConcurrentSet<DependencyItem>();
            unresolved.put(whenRequired, items);
         }
         items.add(item);
      }
   }

   public Set<DependencyItem> getIDependOn(Class<?> type)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn == null)
         iDependOn = Collections.emptySet();
      if (type == null || iDependOn.isEmpty())
         return iDependOn;
      else
      {
         HashSet<DependencyItem> set = new HashSet<DependencyItem>();
         for (DependencyItem item : iDependOn)
         {
            if (type.isInstance(item))
               set.add(item);
         }
         return set;
      }
   }
   
   public void addIDependOn(DependencyItem dependency)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn == null)
      {
         this.iDependOn = new CopyOnWriteArraySet<DependencyItem>();
         iDependOn = this.iDependOn;
      }
      iDependOn.add(dependency);
      flushJBossObjectCache();
      if (dependency instanceof TrackingDependencyItem)
      {
         ((TrackingDependencyItem) dependency).setDependencyInfo(this);
         if (tracking)
         {
            if (dependency.isResolved() == false)
            {
               ControllerState whenRequired = dependency.getWhenRequired();
               Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
               if (unresolved == null)
               {
                  this.unresolved = new ConcurrentHashMap<ControllerState, Set<DependencyItem>>();
                  unresolved = this.unresolved;
               }
               Set<DependencyItem> items = unresolved.get(whenRequired);
               if (items == null)
               {
                  items = new ConcurrentSet<DependencyItem>();
                  unresolved.put(whenRequired, items);
               }
               items.add(dependency);
            }
         }
      }
      else
      {
         if (tracking)
         {
            log.debug("Cannot track " + dependency);
            tracking = false;
            unresolved = null;
         }
      }
   }

   public void removeIDependOn(DependencyItem dependency)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn == null)
         return;
      iDependOn.remove(dependency);
      flushJBossObjectCache();
      if (dependency instanceof TrackingDependencyItem)
      {
         ((TrackingDependencyItem) dependency).setDependencyInfo(null);
         if (tracking)
         {
            ControllerState whenRequired = dependency.getWhenRequired();
            Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
            if (unresolved != null)
            {
               Set<DependencyItem> items = unresolved.get(whenRequired);
               if (items != null)
                  items.remove(dependency);
            }
         }
      }
   }
   
   public Set<DependencyItem> getDependsOnMe(Class<?> type)
   {
      Set<DependencyItem> dependsOnMe = this.dependsOnMe;
      if (dependsOnMe == null)
         dependsOnMe = Collections.emptySet();
      
      if (type == null || dependsOnMe.isEmpty())
         return dependsOnMe;
      else
      {
         HashSet<DependencyItem> set = new HashSet<DependencyItem>();
         for (DependencyItem item : dependsOnMe)
         {
            if (type.isInstance(item))
               set.add(item);
         }
         return set;
      }
   }
   
   public void addDependsOnMe(DependencyItem dependency)
   {
      Set<DependencyItem> dependsOnMe = this.dependsOnMe;
      if (dependsOnMe == null)
      {
         this.dependsOnMe = new CopyOnWriteArraySet<DependencyItem>();
         dependsOnMe = this.dependsOnMe;
      }
      dependsOnMe.add(dependency);
      flushJBossObjectCache();
   }

   public void removeDependsOnMe(DependencyItem dependency)
   {
      Set<DependencyItem> dependsOnMe = this.dependsOnMe;
      if (dependsOnMe == null)
         return;
      dependsOnMe.remove(dependency);
      flushJBossObjectCache();
   }
   
   public boolean resolveDependencies(Controller controller, ControllerState state)
   {
      boolean resolved = true;
      Set<DependencyItem> items = getUnresolvedDependencies(state);
      if (items.isEmpty() == false)
      {
         for (DependencyItem item : items)
         {
            if (item.resolve(controller) == false)
               resolved = false;
         }
      }
      return resolved;
   }

   public Set<DependencyItem> getUnresolvedDependencies(ControllerState state)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn == null || iDependOn.isEmpty())
         return Collections.emptySet();
      
      Set<DependencyItem> result = null;

      if (tracking && state != null)
      {
         Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
         if (unresolved != null)
            result = unresolved.get(state);
         if (result == null)
            result = Collections.emptySet();
         return result;
      }
      
      for (DependencyItem item : iDependOn)
      {
         if (state == null || state.equals(item.getWhenRequired()))
         {
            if (item.isResolved() == false)
            {
               if (result == null)
                  result = new HashSet<DependencyItem>();
               result.add(item);
            }
         }
      }
      if (result == null)
         return Collections.emptySet();
      return result;
   }

   public <T> void addInstallItem(CallbackItem<T> callbackItem)
   {
      Set<CallbackItem<?>> installCallbacks = this.installCallbacks;
      if (installCallbacks == null)
      {
         this.installCallbacks = new CopyOnWriteArraySet<CallbackItem<?>>();
         installCallbacks = this.installCallbacks;
      }
      installCallbacks.add(callbackItem);
      flushJBossObjectCache();
   }

   public <T> void removeInstallItem(CallbackItem<T> callbackItem)
   {
      Set<CallbackItem<?>> installCallbacks = this.installCallbacks;
      if (installCallbacks == null)
         return;
      installCallbacks.remove(callbackItem);
      flushJBossObjectCache();
   }

   public Set<CallbackItem<?>> getInstallItems()
   {
      Set<CallbackItem<?>> installCallbacks = this.installCallbacks;
      if (installCallbacks == null)
         return Collections.emptySet();
      return installCallbacks;
   }

   public <T> void addUninstallItem(CallbackItem<T> callbackItem)
   {
      Set<CallbackItem<?>> uninstallCallbacks = this.uninstallCallbacks;
      if (uninstallCallbacks == null)
      {
         this.uninstallCallbacks = new CopyOnWriteArraySet<CallbackItem<?>>();
         uninstallCallbacks = this.uninstallCallbacks;
      }
      uninstallCallbacks.add(callbackItem);
      flushJBossObjectCache();
   }

   public <T> void removeUninstallItem(CallbackItem<T> callbackItem)
   {
      Set<CallbackItem<?>> uninstallCallbacks = this.uninstallCallbacks;
      if (uninstallCallbacks == null)
         return;
      uninstallCallbacks.remove(callbackItem);
      flushJBossObjectCache();
   }

   public Set<CallbackItem<?>> getUninstallItems()
   {
      Set<CallbackItem<?>> uninstallCallbacks = this.uninstallCallbacks;
      if (uninstallCallbacks == null)
         return Collections.emptySet();
      return uninstallCallbacks;
   }

   public void addLifecycleCallback(LifecycleCallbackItem lifecycleCallbackItem)
   {
      List<LifecycleCallbackItem> lifecycleCallbacks = this.lifecycleCallbacks;
      if (lifecycleCallbacks == null)
      {
         this.lifecycleCallbacks = new CopyOnWriteArrayList<LifecycleCallbackItem>();
         lifecycleCallbacks = this.lifecycleCallbacks;
      }
      lifecycleCallbacks.add(lifecycleCallbackItem);
   }
   
   public List<LifecycleCallbackItem> getLifecycleCallbacks()
   {
      List<LifecycleCallbackItem> lifecycleCallbacks = this.lifecycleCallbacks;
      if (lifecycleCallbacks == null)
         return Collections.emptyList();
      return lifecycleCallbacks;
   }
   
   public boolean isAutowireCandidate()
   {
      return autowireCandidate;
   }

   public void setAutowireCandidate(boolean candidate)
   {
      this.autowireCandidate = candidate;
   }

   public void toString(JBossStringBuilder buffer)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn != null)
         buffer.append("idependOn=").append(iDependOn);
      try
      {
         Set<DependencyItem> unresolved = getUnresolvedDependencies(null);
         if (unresolved.isEmpty() == false)
            buffer.append(" unresolved=").append(unresolved);
      }
      catch (Throwable ignored)
      {
         buffer.append(" unresolved=" + ignored);
      }
   }
}
