/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.dependency.plugins.tracker;

import java.util.*;

import org.jboss.dependency.spi.Controller;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.dependency.spi.ControllerStateModel;
import org.jboss.dependency.spi.tracker.ContextFilter;
import org.jboss.dependency.spi.tracker.ContextQueries;
import org.jboss.dependency.spi.tracker.ContextRegistry;

/**
 * Abstract context registry
 *
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 */
public class AbstractContextRegistry extends AbstractLockHolder implements ContextQueries, ContextRegistry
{
   /** The contexts by class Map<Class, Set<ControllerContext>> */
   private Map<Class<?>, ClassContext> contextsByClass = new HashMap<Class<?>, ClassContext>();

   /** The classes by contexts */
   private Map<ControllerContext, Set<Class<?>>> classesByContext = new HashMap<ControllerContext, Set<Class<?>>>();

   /** The controller */
   private Controller controller;

   public AbstractContextRegistry(Controller controller)
   {
      if (controller == null)
         throw new IllegalArgumentException("Null controller");
      this.controller = controller;
   }

   public ControllerContext getContext(Object name, ControllerState state)
   {
      return controller.getContext(name, state);
   }

   public ControllerContext getInstalledContext(Object name)
   {
      return controller.getInstalledContext(name);
   }

   public Set<ControllerContext> getNotInstalled()
   {
      return controller.getNotInstalled();
   }

   public Set<ControllerContext> getContextsByState(ControllerState state)
   {
      return controller.getContextsByState(state);
   }

   public Set<ControllerContext> filter(Iterable<ControllerContext> contexts, ContextFilter filter)
   {
      if (contexts == null)
         return null;

      Set<ControllerContext> set = new HashSet<ControllerContext>();
      for (ControllerContext context : contexts)
      {
         if (filter == null || filter.accepts(context))
            set.add(context);
      }
      return set;
   }

   /**
    * Get contexts by class.
    * This method should be taken with read lock.
    *
    * @param clazz the class type
    * @return contexts by class
    */
   protected Set<ControllerContext> getContexts(Class<?> clazz)
   {
      ClassContext classContext = contextsByClass.get(clazz);
      if (classContext != null)
      {
         if (log.isTraceEnabled())
         {
            log.trace("Marking class " + clazz + " as used.");
         }
         classContext.used = true;
         return classContext.contexts;
      }
      return null;
   }

   /**
    * Get instantiated contexts.
    *
    * @param clazz the class to match
    * @return all instantiated contexts whose target is instance of this class clazz param
    */
   public Set<ControllerContext> getInstantiatedContexts(Class<?> clazz)
   {
      lockRead();
      try
      {
         Set<ControllerContext> contexts = getContexts(clazz);
         return contexts != null && contexts.isEmpty() == false ? Collections.unmodifiableSet(contexts) : null;
      }
      finally
      {
         unlockRead();
      }
   }

   public Set<ControllerContext> getContexts(Class<?> clazz, ControllerState state)
   {
      lockRead();
      try
      {
         Set<ControllerContext> contexts = getContexts(clazz);
         if (contexts != null && contexts.isEmpty() == false)
         {
            ControllerStateModel model = controller.getStates();
            Set<ControllerContext> kccs = new HashSet<ControllerContext>();
            for(ControllerContext context : contexts)
            {
               if (model.isBeforeState(context.getState(), state) == false)
                  kccs.add(context);
            }
            return kccs;
         }
         else
            return null;
      }
      finally
      {
         unlockRead();
      }
   }

   public void addInstantiatedContext(ControllerContext context)
   {
      prepareToTraverse(context, true);
   }

   public void registerInstantiatedContext(ControllerContext context, Class<?>... classes)
   {
      handleInstantiatedContext(context, true, classes);
   }

   /**
    * Register / unregister contexts against explicit classes.
    *
    * @param context the context
    * @param addition whether this is an addition
    * @param classes the exposed classes
    */
   protected void handleInstantiatedContext(ControllerContext context, boolean addition, Class<?>... classes)
   {
      if (context == null)
         throw new IllegalArgumentException("Null context");

      if (classes != null && classes.length > 0)
      {
         boolean trace = log.isTraceEnabled();

         lockWrite();
         try
         {
            for (Class<?> clazz : classes)
               handleContext(context, clazz, addition, trace);

            if (addition)
            {
               Set<Class<?>> clazzes = new HashSet<Class<?>>(Arrays.asList(classes));
               classesByContext.put(context, clazzes);
            }
            else
            {
               classesByContext.remove(context);
            }
         }
         finally
         {
            unlockWrite();
         }
      }
   }

   public void unregisterInstantiatedContext(ControllerContext context, Class<?>... classes)
   {
      handleInstantiatedContext(context, false, classes);
   }

   public void removeInstantiatedContext(ControllerContext context)
   {
      prepareToTraverse(context, false);
   }

   protected void prepareToTraverse(ControllerContext context, boolean addition)
   {
      lockWrite();
      try
      {
         Object target = context.getTarget();
         if (target != null)
         {
            Set<Class<?>> classes = addition ? new HashSet<Class<?>>() : null;
            traverseBean(context, target.getClass(), addition, classes, log.isTraceEnabled());
            if (addition)
            {
               classesByContext.put(context, classes);
            }
            else
            {
               classesByContext.remove(context);
            }
         }
      }
      finally
      {
         unlockWrite();
      }
   }

   /**
    * Traverse over target and map it to all its superclasses
    * and interfaces - using recursion.
    *
    * @param context context whose target is instance of clazz
    * @param clazz current class to map context to
    * @param addition whether this is an addition
    * @param classes the exposed classes
    * @param trace whether trace is enabled
    */
   protected void traverseBean(ControllerContext context, Class<?> clazz, boolean addition, Set<Class<?>> classes, boolean trace)
   {
      if (clazz == null || clazz == Object.class)
      {
         return;
      }

      handleContext(context, clazz, addition, trace);
      if (addition)
         classes.add(clazz);

      // traverse superclass
      traverseBean(context, clazz.getSuperclass(), addition, classes, trace);
      Class<?>[] interfaces = clazz.getInterfaces();
      // traverse interfaces
      for(Class<?> intface : interfaces)
      {
         traverseBean(context, intface, addition, classes, trace);
      }
   }

   /**
    * Map or remove context against class.
    * This method should be used with write lock taken before.
    *
    * @param context the context
    * @param clazz the class
    * @param addition whether this is an addition
    * @param trace trace is enabled
    */
   protected void handleContext(ControllerContext context, Class<?> clazz, boolean addition, boolean trace)
   {
      ClassContext classContext = contextsByClass.get(clazz);
      if (addition)
      {
         if (classContext == null)
         {
            classContext = new ClassContext();
            classContext.contexts = new HashSet<ControllerContext>();
            contextsByClass.put(clazz, classContext);
         }
         else if (classContext.used)
         {
            log.debugf("Additional matching bean - contextual injection already used for class: %1s", clazz);
         }
         if (trace)
         {
            log.trace("Mapping contex " + context + " to class: " + clazz);
         }
         classContext.contexts.add(context);
      }
      else
      {
         if (classContext != null)
         {
            if (trace)
            {
               log.trace("Removing contex " + context + " to class: " + clazz);
            }
            classContext.contexts.remove(context);
            if (classContext.contexts.isEmpty())
               contextsByClass.remove(clazz);
         }
      }
   }

   /**
    * If zero or multiple instances match class clazz
    * a warning is issued, but no throwable is thrown
    *
    * @param clazz the class to match
    * @return context whose target is instance of this class clazz param or null if zero or multiple such instances
    */
   public ControllerContext getContextByClass(Class<?> clazz)
   {
      Set<ControllerContext> contexts = getInstantiatedContexts(clazz);
      int numberOfMatchingBeans = 0;
      if (contexts != null)
      {
         numberOfMatchingBeans = contexts.size();
      }

      if (log.isTraceEnabled())
      {
         log.trace("Checking for contextual injection, current matches: " + numberOfMatchingBeans + " - " + clazz);
      }

      if (numberOfMatchingBeans != 1)
      {
         if (numberOfMatchingBeans > 1)
         {
            log.warn("Multiple beans match class type [enable trace log for details]: " + clazz);
            if (log.isTraceEnabled())
            {
               log.trace("Matching contexts: " + contexts);
            }
         }
         return null;
      }
      return contexts.iterator().next();
   }

   public Set<Class<?>> getExposedClasses(ControllerContext context)
   {
      lockRead();
      try
      {
         Set<Class<?>> result = classesByContext.get(context);
         return (result == null) ? Collections.<Class<?>>emptySet() : new HashSet<Class<?>>(result);
      }
      finally
      {
         unlockRead();
      }
   }

   private static class ClassContext
   {
      private boolean used;
      private Set<ControllerContext> contexts;
   }
}