/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.dependency.spi.tracker;

import java.util.Set;

import org.jboss.dependency.spi.ControllerContext;

/**
 * Track contexts.
 *
 * @author <a href="ales.justin@jboss.org">Ales Justin</a>
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 */
public interface ContextTracker
{
   /**
    * Count how many times "used" is used by user.
    *
    * @param used the used context
    * @param user the user that tracks used.
    * @return tracking count
    */
   int getUsedByCount(ControllerContext used, Object user);

   /**
    * Add context's user.
    * No actual usage yet.
    *
    * @param used the used context
    * @param user the user that tracks used.
    */
   void addUsedBy(ControllerContext used, Object user);

   /**
    * Increment context's user.
    *
    * @param used the used context
    * @param user the user that tracks used.
    */
   void incrementUsedBy(ControllerContext used, Object user);

   /**
    * Decrement context's user.
    *
    * @param used the used context
    * @param user the user that tracks used.
    */
   void decrementUsedBy(ControllerContext used, Object user);

   /**
    * Remove context's user.
    *
    * @param used the used context
    * @param user the user that tracks used.
    */
   void removeUsedBy(ControllerContext used, Object user);

   /**
    * Is someone using "used"?
    *
    * @param used the context to check
    * @return true if the used context is in use
    */
   boolean isContextInUse(ControllerContext used);

   /**
    * Who all is using "used" context?
    *
    * @param used the context to check
    * @return used users
    */
   Set<Object> getUsers(ControllerContext used);

   /**
    * Which contexts do I use?
    *
    * If null type param is passed,
    * not type check is performed.
    *
    * @param user the user
    * @return all used contexts
    */
   Set<ControllerContext> getUsedContexts(Object user);
}