/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.metadata;

import java.io.Serializable;

import org.jboss.dependency.plugins.AbstractDependencyItem;
import org.jboss.dependency.spi.Controller;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.dependency.spi.DependencyItem;
import org.jboss.dependency.spi.dispatch.AttributeDispatchContext;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.system.microcontainer.ServiceControllerContext;

/**
 * ServiceInjectionValueMetaData.
 * 
 * Creates an MC injection if the element contained in an MBean's <code>attribute</code> element is <code>inject</code>.
 * 
 * <pre>
 *   &lt;mbean name="jboss.test:type=test" code="org.jboss.test.system.controller.support.Simple"&gt;
 *       &lt;attribute name="Other" serialDataType="jbxb"&gt;
 *           &lt;inject bean="SomeBean"/&gt;
 *       &lt;/attribute&gt;   
 *   &lt;/mbean&gt;
 * </pre>
 *  
 * This will inject the bean <code>SomeBean</code> into the MBean's <code>Other</code> property.
 * The value of the <code>inject</code> can either be the ObjectName of another MBean or the
 * name of a bean installed in the underlying {@link Controller}
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.org">Ales Justin</a>
 */
public class ServiceInjectionValueMetaData extends AbstractMetaDataVisitorNode
   implements ServiceValueMetaData, Serializable
{
   private static final long serialVersionUID = 2;

   /** The dependency */
   private Object dependency;

   /** The property */
   private String property;

   /** The required state of the dependency */
   private ControllerState dependentState = ControllerState.INSTALLED;

   /** The underlying context */
   private ServiceControllerContext underlyingContext;

   /**
    * Create a new ServiceInjectionValueMetaData.
    */
   public ServiceInjectionValueMetaData()
   {
   }

   /**
    * Create a new ServiceInjectionValueMetaData.
    * 
    * @param dependency the dependency
    */
   public ServiceInjectionValueMetaData(Object dependency)
   {
      this(dependency, null);
   }
   
   /**
    * Create a new ServiceInjectionValueMetaData.
    * 
    * @param dependency the dependency
    * @param property the property name
    */
   public ServiceInjectionValueMetaData(Object dependency, String property)
   {
      this(dependency, property, ControllerState.INSTALLED);
   }
   
   /**
    * Create a new ServiceInjectionValueMetaData.
    * 
    * @param dependency the dependency
    * @param property the property name
    * @param dependentState the dependent state
    */
   public ServiceInjectionValueMetaData(Object dependency, String property, ControllerState dependentState)
   {
      setDependency(dependency);
      setProperty(property);
      setDependentState(dependentState);
   }

   /**
    * Get the dependency.
    * 
    * @return the dependency.
    */
   public Object getDependency()
   {
      return dependency;
   }

   /**
    * Set the dependency.
    * 
    * @param dependency the dependency.
    */
   public void setDependency(Object dependency)
   {
      if (dependency == null)
         throw new IllegalArgumentException("Null dependency");
      this.dependency = dependency;
   }

   /**
    * Get the property.
    * 
    * @return the property.
    */
   public String getProperty()
   {
      return property;
   }

   /**
    * Set the property.
    * 
    * @param property the property.
    */
   public void setProperty(String property)
   {
      this.property = property;
   }

   /**
    * Get the dependentState.
    * 
    * @return the dependentState.
    */
   public ControllerState getDependentState()
   {
      return dependentState;
   }

   /**
    * Set the dependentState.
    * 
    * @param dependentState the dependentState.
    */
   public void setDependentState(ControllerState dependentState)
   {
      this.dependentState = dependentState;
   }

   public Object getValue(ServiceValueContext valueContext) throws Throwable
   {
      KernelController controller = valueContext.getController();
      
      ControllerState state = dependentState;
      if (state == null)
         state = ControllerState.INSTALLED;

      ControllerContext context = controller.getContext(dependency, state);
      if (context == null)
         throw new Error("Should not be here - dependency failed! " + this);

      if (property != null && property.length() > 0)
      {
         if (context instanceof AttributeDispatchContext)
         {
            AttributeDispatchContext adc = (AttributeDispatchContext)context;
            return adc.get(property);
         }
         else
            throw new IllegalArgumentException(
                  "Cannot use property attribute, context is not AttributeDispatchContext: " + context +
                  ", metadata: " + this);
      }
      else
      {
         return getTarget(underlyingContext, context);
      }
   }

   @Override
   public Object ungetValue(ServiceValueContext valueContext) throws Throwable
   {
      if (property == null || property.length() == 0)
      {
         KernelController controller = valueContext.getController();

         ControllerState state = dependentState;
         if (state == null)
            state = ControllerState.INSTALLED;

         ControllerContext context = controller.getContext(dependency, state);
         if (context != null)
         {
            ungetTarget(underlyingContext, context);
            return null;
         }
      }
      return super.ungetValue(valueContext);
   }

   public void visit(ServiceMetaDataVisitor visitor)
   {
      ServiceControllerContext context = visitor.getControllerContext();
      underlyingContext = context; // remember context, leak?
      Object name = context.getName();
      ControllerState whenRequired = visitor.getContextState();

      DependencyItem item = new AbstractDependencyItem(name, dependency, whenRequired, dependentState);
      visitor.addDependency(item);

      visitor.visit(this);
   }
}
