/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.spi.qualifier;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.jboss.beans.metadata.api.model.QualifierContent;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.kernel.plugins.qualifier.AnnotationQualiferParser;
import org.jboss.kernel.plugins.qualifier.DefaultEqualsMatcher;
import org.jboss.kernel.plugins.qualifier.StringQualifierParser;

/**
 * Singleton registry of qualifier matchers and parsers, and the entry point to the qualifier matcher subsystem
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class QualifierMatchers
{
   private final static QualifierMatchers SINGLETON = new QualifierMatchers();

   //Specify concurrency level = 1 since it is very unlikely more than one thread will be adding matchers at any time
   private ConcurrentMap<Class<?>, QualifierMatcher> matchers = new ConcurrentHashMap<Class<?>, QualifierMatcher>(3, .75f, 1);
   
   //Specify concurrency level = 1 since it is very unlikely more than one thread will be adding parsers at any time
   private ConcurrentMap<QualifierContent, QualifierParser> parsers = new ConcurrentHashMap<QualifierContent, QualifierParser>(3, .75f, 1);
   
   private QualifierMatchers()
   {
      addParser(StringQualifierParser.INSTANCE);
      addParser(AnnotationQualiferParser.INSTANCE);
   }
   
   /**
    * Get the singleton instance
    * 
    * @return the singleton instance
    */
   public static QualifierMatchers getInstance()
   {
      return SINGLETON;
   }
   
   /**
    * Get the qualifier parser to be used for a content type
    * 
    * @param content the content type
    * @return the parser
    */
   public QualifierParser getParser(QualifierContent content)
   {
      QualifierParser parser = parsers.get(content);
      if (parser == null)
         throw new IllegalArgumentException("No parser for " + content);
      
      return parser;
   }
   
   /**
    * Check whether the passed in qualifier matches the candidate context and/or the context's supplied qualifiers.
    * If there is no matcher for the qualifiers type a default implementation is used. The default implementation
    * iterates over the set of supplied qualifiers and returns true if any of them match the qualifier using Object#equals(Object).
    * 
    * @param context the context to check if matches the passed in qualifier
    * @param suppliedQualifiers the supplied qualifiers for the context
    * @param qualifier the qualifier to check against the context and/or its qualifiers
    * @return true if there is a match
    */
   @SuppressWarnings({"unchecked"})
   public boolean matches(ControllerContext context, Set<Object> suppliedQualifiers, Object qualifier)
   {
      Class<?> clazz = qualifier.getClass();
      QualifierMatcher matcher = null;
      for (Map.Entry<Class<?>, QualifierMatcher> entry : matchers.entrySet())
      {
         if (entry.getKey().isAssignableFrom(clazz))
         {
            matcher = entry.getValue();
            break;
         }
      }
      if (matcher == null)
         matcher = DefaultEqualsMatcher.SINGLETON; 
      return matcher.matches(context, suppliedQualifiers, qualifier);
   }
   
   /**
    * Add a parser. It will be used to parse qualifiers with the content type {@link QualifierParser#getHandledContent()}
    * 
    * @param parser the parser to add
    * @throws IllegalArgumentException if a parser of a different type already exists for the handled content type
    */
   public void addParser(QualifierParser parser)
   {
      QualifierParser old = parsers.putIfAbsent(parser.getHandledContent(), parser);
      
      if (old != null && old.getClass() != parser.getClass())
         throw new IllegalArgumentException(old + " is already handling " + parser.getHandledContent());
   }

   /**
    * Remove a parser for the given qualifier content
    * 
    * @param content the qualifier content we want to remove a parser for
    */
   public void removeParser(QualifierContent content)
   {
      parsers.remove(content);
   }
   
   /**
    * Add a matcher. It is used when comparing a wanted qualifier against a candidate
    * context and its supplied qualifiers. The matcher handles wanted qualifiers of
    * the type {@link QualifierMatcher#getHandledType()}.
    * 
    *  @param matcher the matcher to add
    *  @throwa IllegalArgumentException if a matcher of a different type already exists for the handled qualifier type
    */
   public void addMatcher(QualifierMatcher matcher)
   {
      QualifierMatcher old = matchers.putIfAbsent(matcher.getHandledType(), matcher);
      
      if (old != null && old.getClass() != matcher.getClass())
         throw new IllegalArgumentException(old + " is already handling " + matcher.getHandledType());
   }

   /**
    * Remove any matcher for the given qualifier type
    * 
    * @param clazz the qualifier type we want to remove the matcher for
    */
   public void removeMatcher(Class<?> clazz)
   {
      matchers.remove(clazz);
   }
}
