/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.beans.metadata.api.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;


/**
 * Used to specify a bean's demand.
 * 
 * <pre>
 * &#64;Bean(name="SomeBean")
 * &#64;Demand("java:/something")
 * public class MyBean
 * {
 * }
 * </pre>
 * <code>SomeBean</code> cannot be installed until another bean called, or another bean that supplies something
 * called <code>java:/something</code> has been installed.  
 *  
 * @see Supplys
 * @see Supply
 * @see Demands
 * @see org.jboss.beans.metadata.spi.BeanMetaData#getDemands()
 * @author <a href="mailto:ales.justin@jboss.com">Ales Justin</a>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.ANNOTATION_TYPE})
public @interface Demand
{
   /**
    * Get demand value.
    *
    * @return the name of what we are demanding
    */
   String value();

   /**
    * Get when required. The default is the {@link org.jboss.dependency.spi.ControllerState#INSTANTIATED} state
    *
    * @return the when required state
    */
   String whenRequired() default "Instantiated";

   /**
    * Get the transfomer, which is used to determine the {@link org.jboss.kernel.api.dependency.Matcher} to 
    * use via the {@link org.jboss.kernel.api.dependency.MatcherFactory}. The Matcher determines how to match the
    * names of the available supplies, with the demand value.
    *
    * @return the transformer
    */
   String transformer() default "";
}
