/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.beans.metadata.plugins;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import org.jboss.beans.metadata.api.model.AutowireType;
import org.jboss.beans.metadata.api.model.FromContext;
import org.jboss.beans.metadata.api.model.InjectOption;
import org.jboss.beans.metadata.api.model.QualifierPoint;
import org.jboss.beans.metadata.spi.ConstructorMetaData;
import org.jboss.beans.metadata.spi.LifecycleMetaData;
import org.jboss.beans.metadata.spi.MetaDataVisitor;
import org.jboss.beans.metadata.spi.MetaDataVisitorNode;
import org.jboss.beans.metadata.spi.PropertyMetaData;
import org.jboss.beans.metadata.spi.RelatedClassMetaData;
import org.jboss.dependency.plugins.AttributeCallbackItem;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.kernel.plugins.dependency.ClassAndQualifiersKey;
import org.jboss.kernel.plugins.dependency.QualifierKey;
import org.jboss.kernel.plugins.dependency.QualifiersMdrUtil;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.JBossStringBuilder;

/**
 * Injection value.
 *
 * @author <a href="ales.justin@gmail.com">Ales Justin</a>
 */
@XmlType(name="injectionType", propOrder="qualifiers")
public class AbstractInjectionValueMetaData extends AbstractDependencyValueMetaData
{
   private static final long serialVersionUID = 3L;

   protected AutowireType injectionType = AutowireType.BY_CLASS;

   protected InjectOption injectionOption = InjectOption.STRICT;

   protected FromContext fromContext;
   
   ContextualInjectionDependencyItem item = null;

   /** If qualifiers are used, true if we should ignore the qualifiers on the bean and only take into account the qualifiers on this injection point*/ 
   protected boolean ignoreBeanQualifiers;
   
   /** Qualifiers specified on the injection point. If not present the context's wanted qualifiers from the MDR metadata is used */
   protected Set<RelatedClassMetaData> qualifiers;
   
   /** The parent nodes, with the lowest node first */
   protected List<MetaDataVisitorNode> parentNodes;

   /**
    * Simplyifies things with AutowireType.BY_NAME
    */
   protected AbstractPropertyMetaData propertyMetaData;
   
   /**
    * Create a new injection value
    */
   public AbstractInjectionValueMetaData()
   {
   }

   /**
    * Create a new injection value
    *
    * @param value the value
    */
   public AbstractInjectionValueMetaData(Object value)
   {
      super(value);
   }

   /**
    * Create a new injection value
    *
    * @param value    the value
    * @param property the property
    */
   public AbstractInjectionValueMetaData(Object value, String property)
   {
      super(value, property);
   }

   public AutowireType getInjectionType()
   {
      return injectionType;
   }

   @XmlAttribute(name="type")
   public void setInjectionType(AutowireType injectionType)
   {
      this.injectionType = injectionType;
   }

   public InjectOption getInjectionOption()
   {
      return injectionOption;
   }

   @XmlAttribute(name="option")
   public void setInjectionOption(InjectOption injectionOption)
   {
      this.injectionOption = injectionOption;
   }

   public FromContext getFromContext()
   {
      return fromContext;
   }

   @XmlAttribute(name="fromContext")
   public void setFromContext(FromContext fromContext)
   {
      this.fromContext = fromContext;
   }

   public AbstractPropertyMetaData getPropertyMetaData()
   {
      return propertyMetaData;
   }

   @XmlTransient
   public void setPropertyMetaData(AbstractPropertyMetaData propertyMetaData)
   {
      this.propertyMetaData = propertyMetaData;
   }
   
   /**
    * Get the qualifiers
    * @return the qualifiers
    */
   public Set<RelatedClassMetaData> getQualifiers()
   {
      return qualifiers;
   }

   /**
    * Set the qualifiers
    * @param qualifiers the qualifiers to set
    */
   @XmlElement(name="qualifier", type=AbstractInjectQualifierMetaData.class)
   public void setQualifiers(Set<RelatedClassMetaData> qualifiers)
   {
      this.qualifiers = qualifiers;
   }

   /**
    * Get the ignoreBeanQualifiers
    * @return the ignoreBeanQualifiers
    */
   public boolean isIgnoreBeanQualifiers()
   {
      return ignoreBeanQualifiers;
   }

   /**
    * Set the ignoreBeanQualifiers
    * @param ignoreBeanQualifiers the ignoreBeanQualifiers to set
    */
   @XmlAttribute(name="ignoreBeanQualifiers")
   public void setIgnoreBeanQualifiers(boolean ignoreBeanQualifiers)
   {
      this.ignoreBeanQualifiers = ignoreBeanQualifiers;
   }

   /**
    * Add install/callback item.
    *
    * @param name the callback name
    */
   protected void addInstallItem(Object name)
   {
      if (propertyMetaData == null)
         throw new IllegalArgumentException("Illegal usage of option Callback - injection not used with property = " + this);
      context.getDependencyInfo().addInstallItem(new AttributeCallbackItem<Object>(name, whenRequiredState, dependentState, context, propertyMetaData.getName()));
   }

   protected boolean isLookupValid(ControllerContext lookup)
   {
      boolean lookupExists = super.isLookupValid(lookup);
      boolean isCallback = InjectOption.CALLBACK.equals(injectionOption);
      boolean isOptional = InjectOption.OPTIONAL.equals(injectionOption);
      if (lookupExists == false && isCallback)
      {
         addInstallItem(getUnderlyingValue());
      }
      return lookupExists || isCallback || isOptional;
   }

   protected boolean isOptional()
   {
      return InjectOption.OPTIONAL.equals(injectionOption);
   }

   @SuppressWarnings({"deprecation"})
   public Object getValue(TypeInfo info, ClassLoader cl) throws Throwable
   {
      // controller context property injection
      if (fromContext != null)
      {
         ControllerState state = dependentState;
         if (state == null)
            state = ControllerState.INSTANTIATED;

         ControllerContext lookup = getControllerContext(getUnderlyingValue(), state);
         if (lookup == null)
            throw new Error("Should not be here - dependency failed - " + this);

         return fromContext.executeLookup(lookup);
      }

      // by class type
      if (getUnderlyingValue() == null)
      {
         ControllerContext lookup = null;
         if (item != null)
            lookup = item.getControllerContext(context.getController());

         if (lookup == null)
         {
            if (InjectOption.STRICT.equals(injectionOption))
            {
               throw new IllegalArgumentException("Possible multiple matching beans, see log for info.");
            }
            else
            {
               if (InjectOption.CALLBACK.equals(injectionOption))
                  addInstallItem(info.getType());

               return null;
            }
         }

         // TODO - add progression here, then fix BeanMetaData as well
         return getTarget(context, lookup);
      }
      return super.getValue(info, cl);
   }

   @SuppressWarnings({"deprecation"})
   public Object ungetValue(TypeInfo info, ClassLoader cl) throws Throwable
   {
      if (getUnderlyingValue() == null)
      {
         ControllerContext lookup = item.getControllerContext(context.getController());
         if (lookup != null)
         {
            ungetTarget(context, lookup);
            return null;
         }
      }
      return super.ungetValue(info, cl);
   }

   protected boolean addDependencyItem()
   {
      return InjectOption.STRICT.equals(injectionOption) || fromContext != null;
   }

   @XmlTransient
   public Object getUnderlyingValue()
   {
      Object original = super.getUnderlyingValue();
      // might be used for internal compare, in that case context will still be null
      return (fromContext != null && original == null) ? (context != null ? context.getName() : null) : original;
   }

   public void initialVisit(MetaDataVisitor visitor)
   {
      // controller context property injection
      if (fromContext != null)
      {
         // check if dependent is not set when used on itself
         if (super.getUnderlyingValue() == null && dependentState == null)
         {
            dependentState = fromContext.getWhenValid();
         }

         super.initialVisit(visitor);
         return;
      }

      // no bean specified
      if (getUnderlyingValue() == null)
      {
         // check for property
         if (property != null)
         {
            property = null;
            log.warn("Ignoring property - contextual injection: " + this);
         }

         if (AutowireType.BY_NAME.equals(injectionType))
         {
            if (propertyMetaData == null)
               throw new IllegalArgumentException("Illegal usage of type ByName - injection not used with property = " + this);
            setValue(propertyMetaData.getName());
         }
         else
         {
            visitor.initialVisit(this);
         }
      }
      // check if was maybe set with by_name
      if (getUnderlyingValue() != null)
      {
         super.initialVisit(visitor);
      }
   }

   @SuppressWarnings("deprecation")
   public void describeVisit(MetaDataVisitor visitor)
   {
      // no bean and not by_name
      if (getUnderlyingValue() == null)
      {
         if (AutowireType.BY_CLASS.equals(injectionType))
         {
            context = visitor.getControllerContext();

            // dependency item or install item
            if (InjectOption.STRICT.equals(injectionOption))
            {
               // add dependency item only for strict inject behaviour
               // we pop it so that parent node has the same semantics as this one
               // meaning that his current peek is also his parent
               // and all other nodes that cannot determine type follow the same
               // contract - popping and pushing
               // maybe the whole thing can be rewritten to LinkedList
               // or simply using the fact that Stack is also a Vector?
               MetaDataVisitorNode node = visitor.visitorNodeStack().pop();
               try
               {
                  if (node instanceof TypeProvider)
                  {
                     TypeProvider typeProvider = (TypeProvider)node;
                     Class<?> injectionClass = null;
                     try
                     {
                        injectionClass = typeProvider.getType(visitor, this).getType();
                     }
                     finally
                     {
                        visitor.visitorNodeStack().push(node);
                     }
                     log.debugf("%1s : Contextual injection usage (class -> classloader): %2s -> %3s defined by %4s", context.getName(), injectionClass, SecurityActions.getClassLoader(injectionClass), node);
                     
                     // set when required
                     ControllerState whenRequired = whenRequiredState;
                     if (whenRequired == null)
                     {
                        whenRequired = visitor.getContextState();
                     }
                     item = createDependencyItem(visitor, injectionClass, whenRequired);
                     visitor.addDependency(item);
                  }
                  else
                  {
                     throw new Error(TypeProvider.ERROR_MSG);
                  }
               }
               catch (Error error)
               {
                  throw error;
               }
               catch (Throwable throwable)
               {
                  throw new Error(throwable);
               }
            }
         }
         else
         {
            throw new IllegalArgumentException("Unknown injection type=" + injectionType);
         }
      }
      super.describeVisit(visitor);
   }

   private ContextualInjectionDependencyItem createDependencyItem(MetaDataVisitor visitor, Class<?> injectionClass, ControllerState whenRequired) throws Exception
   {
      initializeParents(visitor);
      QualifierPoint point = determineQualifierPoint();
      
      Set<Object> allQualifiers = getAllQualifiers();
      boolean hasQualifiers = allQualifiers != null && allQualifiers.size() > 0;
      if (!hasQualifiers && point != null)
      {
         hasQualifiers = QualifiersMdrUtil.hasWantedQualifiersInParentMdrOrBeanMetaData(context, point);
      }

      if (!hasQualifiers)
         return new ContextualInjectionDependencyItem(this, context.getName(), injectionClass, whenRequired, dependentState, search);
      else
      {
         QualifierKey qualifierKey = createClassAndQualifierMatcher(injectionClass, allQualifiers);
         return new ContextualInjectionDependencyItem(this, context.getName(), qualifierKey, whenRequired, dependentState, search);
      }
   }

   private Set<Object> getAllQualifiers() throws Exception
   {
      Set<Annotation> qualifierAnnotations = null;
      if (context.getBeanInfo() != null)
      {
         qualifierAnnotations = QualifiersMdrUtil.getQualifiersFromAnnotationsForInjectionPointParents(context, parentNodes);
      }
      
      Set<Object> allQualifiers = null;
      if ((qualifierAnnotations != null && qualifierAnnotations.size() > 0) || (qualifiers != null && qualifiers.size() > 0))
      {
         allQualifiers = new HashSet<Object>();
         if (qualifiers!= null && qualifiers.size() > 0)
         {
            allQualifiers = new HashSet<Object>(qualifiers.size());
            for (RelatedClassMetaData rcmd : qualifiers)
               allQualifiers.addAll(rcmd.getEnabled());            
         }

         if (qualifierAnnotations != null && qualifierAnnotations.size() > 0)
            allQualifiers.addAll(qualifierAnnotations);
      }
      return allQualifiers;
   }
   
   private void initializeParents(MetaDataVisitor visitor)
   {
      parentNodes = new ArrayList<MetaDataVisitorNode>();
      try
      {
         MetaDataVisitorNode node = visitor.visitorNodeStack().pop();
         parentNodes.add(node);
         while (node != null && visitor.visitorNodeStack().size() > 0)
         {
            parentNodes.add(visitor.visitorNodeStack().pop());
         }
      }
      finally
      {
         for (int i = parentNodes.size() - 1 ; i >= 0 ; i--)
            visitor.visitorNodeStack().push(parentNodes.get(i));
      }
   }
   
   /** 
    * Get the parent nodes, with the lowest node first 
    *
    * @return the parent nodes 
    */
   List<MetaDataVisitorNode> getParents()
   {
      return parentNodes;
   }
   
   /**
    * Create a class and qualifier key for qualified contextual injection
    * 
    * @param injectionClass the class
    * @param allQualifiers all qualifiers
    * @return the created key
    * @throws Exception for any error
    */
   QualifierKey createClassAndQualifierMatcher(Class<?> injectionClass, Set<Object> allQualifiers) throws Exception
   {
      QualifierPoint point = determineQualifierPoint();
      if (point == null)
         throw new IllegalArgumentException("Null qualifier point for " + context + ". Parents: " + parentNodes);

      return new ClassAndQualifiersKey(dependentState, context, ignoreBeanQualifiers, point, parentNodes, allQualifiers, injectionClass);
   }
   
   /**
    * Determine the qualifier point type from the list of parents
    * 
    * @return the qualifer point type
    */
   QualifierPoint determineQualifierPoint()
   {
      for (MetaDataVisitorNode node : parentNodes)
      {
         if (node instanceof ConstructorMetaData)
            return QualifierPoint.CONSTRUCTOR;
         else if (node instanceof LifecycleMetaData)
            return QualifierPoint.METHOD;
         else if (node instanceof PropertyMetaData)
            return QualifierPoint.PROPERTY;
      }
      return null;
   }
   
   public void toString(JBossStringBuilder buffer)
   {
      super.toString(buffer);
      if (injectionType != null)
         buffer.append(" injectionType=").append(injectionType);
      if (propertyMetaData != null)
         buffer.append(" propertyMetaData=").append(propertyMetaData.getName()); //else overflow - indefinite recursion
      if (fromContext != null)
         buffer.append(" fromContext=").append(fromContext);
   }

   public AbstractInjectionValueMetaData clone()
   {
      return (AbstractInjectionValueMetaData)super.clone();
   }
}
