/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.beans.metadata.plugins;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Set;

import org.jboss.beans.metadata.spi.MetaDataVisitorNode;
import org.jboss.dependency.plugins.AbstractDependencyItem;
import org.jboss.dependency.spi.Controller;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.dependency.spi.graph.LookupStrategy;
import org.jboss.dependency.spi.graph.SearchInfo;
import org.jboss.kernel.plugins.dependency.ClearableDependencyItem;
import org.jboss.kernel.plugins.dependency.QualifierKey;

/**
 * A dependency item for looking up by class, or by class and qualifiers
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class ContextualInjectionDependencyItem extends AbstractDependencyItem implements ClearableDependencyItem
{
   private LookupType type;
   
   private final SearchInfo search;
   
   private final AbstractInjectionValueMetaData injectionValueMetaData;

   public ContextualInjectionDependencyItem(AbstractInjectionValueMetaData injectionValueMetaData, Object name, Class<?> demandClass, ControllerState whenRequired, ControllerState dependentState, SearchInfo search)
   {
      super(name, demandClass, whenRequired, dependentState);
      this.type = LookupType.CLASS;
      this.search = search;
      this.injectionValueMetaData = injectionValueMetaData;
   }

   public ContextualInjectionDependencyItem(AbstractInjectionValueMetaData injectionValueMetaData, Object name, QualifierKey key, ControllerState whenRequired, ControllerState dependentState, SearchInfo search)
   {
      super(name, key, whenRequired, dependentState);
      this.type = LookupType.KEY;
      this.search = search;
      this.injectionValueMetaData = injectionValueMetaData;
   }

   public void addQualifierAnnotations(Set<Annotation> qualifiers) throws Exception
   {
      if (type == LookupType.CLASS)
      {
         super.setIDependOn(injectionValueMetaData.createClassAndQualifierMatcher((Class<?>)getIDependOn(), null));
      }
      Object iDependOn = getIDependOn();
      if (iDependOn instanceof QualifierKey)
      {
         QualifierKey qk = (QualifierKey)iDependOn;
         qk.addQualifiersFromAnnotations(qualifiers);
      }
      else if (log.isTraceEnabled())
      {
         log.trace("Ignored non qualifier iDependOn instance: " + iDependOn);
      }
   }
   
   /**
    * Get controller context.
    *
    * @param controller the controller
    * @return controller context
    */
   protected ControllerContext getControllerContext(Controller controller)
   {
      LookupStrategy strategy = search.getStrategy();
      return strategy.getContext(controller, getIDependOn(), ControllerState.INSTALLED);
   }

   public List<MetaDataVisitorNode> getParents()
   {
      return injectionValueMetaData.getParents();
   }
   
   /**
    * Get the injectionValueMetaData
    * @return the injectionValueMetaData
    */
   public AbstractInjectionValueMetaData getInjectionValueMetaData()
   {
      return injectionValueMetaData;
   }

   /**
    * Get the search
    * @return the search
    */
   public SearchInfo getSearch()
   {
      return search;
   }
   
   private enum LookupType
   {
      CLASS, KEY
   }
   
   /**
    * Clear the class reference.
    *  
    * @param controller the controller
    */
   public void clear(Controller controller)
   {
      setIDependOn(null);
   }
}