/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.kernel.plugins.dependency;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.SupplyMetaData;
import org.jboss.dependency.plugins.ScopedController;
import org.jboss.dependency.plugins.action.ControllerContextAction;
import org.jboss.dependency.spi.Controller;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.api.dependency.Matcher;
import org.jboss.kernel.plugins.event.AbstractEventEmitter;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.kernel.spi.event.KernelEvent;
import org.jboss.kernel.spi.event.KernelEventFilter;
import org.jboss.kernel.spi.event.KernelEventListener;
import org.jboss.kernel.spi.registry.KernelRegistryEntry;
import org.jboss.kernel.spi.registry.KernelRegistryPlugin;

/**
 * Abstract Kernel controller.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 * @version $Revision: 100812 $
 */
@SuppressWarnings("deprecation")
public class AbstractKernelController extends ScopedController implements KernelController, KernelRegistryPlugin
{
   /** The kernel */
   protected Kernel kernel;

   /** The emitter delegate */
   protected AbstractEventEmitter emitterDelegate = createEventEmitter();

   /** The supplies */
   protected ConcurrentMap<Object, List<KernelControllerContext>> suppliers = new ConcurrentHashMap<Object, List<KernelControllerContext>>();

   /**
    * Create an abstract kernel controller
    * 
    * @throws Exception for any error
    */
   public AbstractKernelController() throws Exception
   {
   }

   protected Map<ControllerState, ControllerContextAction> createAliasActions()
   {
      Map<ControllerState, ControllerContextAction> map = new HashMap<ControllerState, ControllerContextAction>(super.createAliasActions());
      map.put(ControllerState.PRE_INSTALL, InstallScopeAction.INSTANCE);
      return map;
   }

   /**
    * Create event emitter.
    *
    * @return new abstract event emitter instance
    */
   protected AbstractEventEmitter createEventEmitter()
   {
      return new AbstractEventEmitter();
   }

   public KernelControllerContext install(BeanMetaData metaData) throws Throwable
   {
      return install(metaData, null);
   }

   public KernelControllerContext install(BeanMetaData metaData, Object target) throws Throwable
   {
      KernelControllerContext context = new AbstractKernelControllerContext(null, metaData, target);
      install(context);
      return context;
   }

   @SuppressWarnings("unchecked")
   public KernelRegistryEntry getEntry(Object name)
   {
      List<KernelControllerContext> list;
      if (name instanceof Matcher)
         list = matchSupplies((Matcher)name);
      else
         list = suppliers.get(name);

      if (list != null && list.isEmpty() == false)
         return list.get(0);
      else
         return null;
   }

   /**
    * Try matching supplies.
    *
    * @param matcher the matcher
    * @return list of context's who have a matching supply
    */
   protected List<KernelControllerContext> matchSupplies(Matcher matcher)
   {
      List<KernelControllerContext> list = null;
      for(Map.Entry<Object, List<KernelControllerContext>> entry : suppliers.entrySet())
      {
         if (matcher.match(entry.getKey()))
         {
            if (matcher.needExactMatch() == false)
               return entry.getValue();
            else
            {
               if (list != null)
                  throw new IllegalArgumentException("Matcher " + matcher + " only takes exact match, but found second matching supplier.");
               else
                  list = entry.getValue();
            }
         }
      }
      return list;
   }

   @Override
   public ControllerContext getContextInternal(Object name, ControllerState state, boolean enableOnDemand, Controller initiatingController)
   {
      ControllerContext context = super.getContextInternal(name, state, enableOnDemand, initiatingController);
      if (context != null)
         return context;

      // simple qualifier key
      if (name instanceof QualifierKey)
      {
         QualifierKey qm = (QualifierKey)name;
         return qm.search(this);
      }

      if (state == null || ControllerState.INSTALLED.equals(state) || getStates().isAfterState(state, ControllerState.INSTALLED))
      {
         org.jboss.kernel.spi.registry.KernelRegistry registry = kernel.getRegistry();
         try
         {
            return registry.findEntry(name);
         }
         catch (Throwable ignored)
         {
         }
      }
      return null;
   }
   
   public void addSupplies(KernelControllerContext context)
   {
      BeanMetaData metaData = context.getBeanMetaData();
      Set<SupplyMetaData> supplies = metaData.getSupplies();
      if (supplies != null)
      {
         boolean trace = log.isTraceEnabled();

         if (supplies.isEmpty() == false)
         {
            lockWrite();
            try
            {
               for (SupplyMetaData supplied : supplies)
               {
                  Object supply = supplied.getSupply();
                  List<KernelControllerContext> list = suppliers.get(supply);
                  if (list == null)
                  {
                     list = new CopyOnWriteArrayList<KernelControllerContext>();
                     List<KernelControllerContext> old = suppliers.putIfAbsent(supply, list);
                     if (old != null)
                        list = old;
                  }
                  list.add(context);
                  if (trace)
                     log.trace("Suppliers of " + supply + ": " + list);
               }
            }
            finally
            {
               unlockWrite();
            }
         }
      }
   }

   public void removeSupplies(KernelControllerContext context)
   {
      BeanMetaData metaData = context.getBeanMetaData();
      Set<SupplyMetaData> supplies = metaData.getSupplies();
      if (supplies != null)
      {
         boolean trace = log.isTraceEnabled();

         if (supplies.isEmpty() == false)
         {
            lockWrite();
            try
            {
               for (SupplyMetaData supplied : supplies)
               {
                  Object supply = supplied.getSupply();
                  List<KernelControllerContext> list = suppliers.get(supply);
                  if (list != null)
                  {
                     list.remove(context);
                     if (list.isEmpty())
                        suppliers.remove(supply);
                     if (trace)
                        log.trace("Suppliers of " + supply  + ": " + list);
                  }
               }
            }
            finally
            {
               unlockWrite();
            }
         }
      }
   }

   public Kernel getKernel()
   {
      Kernel.checkAccess();
      return kernel;
   }

   public void setKernel(Kernel kernel) throws Throwable
   {
      Kernel.checkConfigure();
      this.kernel = kernel;
   }

   public void fireKernelEvent(KernelEvent event)
   {
      emitterDelegate.fireKernelEvent(event);
   }

   public void registerListener(KernelEventListener listener, KernelEventFilter filter, Object handback) throws Throwable
   {
      emitterDelegate.registerListener(listener, filter, handback);
   }

   public void unregisterListener(KernelEventListener listener, KernelEventFilter filter, Object handback) throws Throwable
   {
      emitterDelegate.unregisterListener(listener, filter, handback);
   }
}
