/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.kernel.plugins.annotations;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;

import org.jboss.beans.metadata.api.annotations.Constructor;
import org.jboss.beans.metadata.plugins.AbstractBeanMetaData;
import org.jboss.beans.metadata.plugins.AbstractConstructorMetaData;
import org.jboss.beans.metadata.plugins.AbstractInjectionValueMetaData;
import org.jboss.beans.metadata.plugins.AbstractParameterMetaData;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.MetaDataVisitorNode;
import org.jboss.beans.metadata.spi.ParameterMetaData;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.metadata.spi.MetaData;
import org.jboss.reflect.spi.ConstructorInfo;
import org.jboss.reflect.spi.ParameterInfo;

/**
 * Constructor annotation plugin.
 *
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class Jsr330InjectConstructorAnnotationPlugin extends AbstractConstructorParameterAnnotationPlugin<Inject, AbstractConstructorMetaData>
{
   public Jsr330InjectConstructorAnnotationPlugin(Map<Class<? extends Annotation>, Annotation2ValueMetaDataAdapter<? extends Annotation>> adapters)
   {
      super(Inject.class, adapters);
   }

   protected boolean checkAnnotatedInfo(ElementType type)
   {
      return ElementType.CONSTRUCTOR == type;
   }

   protected boolean isMetaDataAlreadyPresent(ConstructorInfo info, Constructor annotation, BeanMetaData beanMetaData)
   {
      return beanMetaData.getConstructor() != null;
   }

   protected ParameterInfo[] getParameters(ConstructorInfo info)
   {
      return info.getParameters();
   }

   protected List<MetaDataVisitorNode> handleParameterlessInfo(ConstructorInfo info, Inject annotation, BeanMetaData beanMetaData)
   {
      // do nothing, default will be used
      return null;
   }

   @Override
   protected List<? extends MetaDataVisitorNode> internalApplyAnnotation(ConstructorInfo info, MetaData retrieval, Inject annotation, KernelControllerContext context) throws Throwable
   {
      if (!Jsr330InjectFilter.INSTANCE.handleInject(retrieval))
         return null;

      //Add the standard MC @Inject annotation to each parameter so that the AbstractInjectionValueMetaData gets created
      ParameterInfo[] params = info.getParameters();
      if (params.length == 0)
         return null;
      
      if (context.getBeanMetaData() instanceof AbstractBeanMetaData == false)
         throw new IllegalStateException("Bean meta data is not AbstractBeanMetaData");

      AbstractBeanMetaData beanMetaData = (AbstractBeanMetaData)context.getBeanMetaData();
      
      AbstractConstructorMetaData cmd = null;
      if (beanMetaData.getConstructor() == null)
      {
         cmd = new AbstractConstructorMetaData();
         beanMetaData.setConstructor(cmd);
      }
      else
      {
         if (beanMetaData.getConstructor() instanceof AbstractConstructorMetaData == false)   
            throw new IllegalStateException("Constructor is not AbstractConstructorMetaData");
         else
            cmd = (AbstractConstructorMetaData)beanMetaData.getConstructor();
      }
         
         
      List<ParameterMetaData> parameters = cmd.getParameters();
      if (parameters == null)
      {
         parameters = new ArrayList<ParameterMetaData>();
         cmd.setParameters(parameters);
      }

      for (int i = 0 ; i < params.length ; i++)
      {
         AbstractInjectionValueMetaData inject = new AbstractInjectionValueMetaData();
         AbstractParameterMetaData param = new AbstractParameterMetaData(params[i].getParameterType().getName(), inject);
         parameters.add(param);
      }
      
      return Collections.singletonList(cmd);
   }
   
   protected void setParameterizedMetaData(AbstractConstructorMetaData parameterizedMetaData, BeanMetaData beanMetaData)
   {
      //Just override to satisfy the interface requirements
   }
}
