/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.weld.plugins.annotated;

import java.util.HashSet;
import java.util.Set;

import javax.enterprise.inject.spi.AnnotatedConstructor;
import javax.enterprise.inject.spi.AnnotatedField;
import javax.enterprise.inject.spi.AnnotatedMethod;
import javax.enterprise.inject.spi.AnnotatedType;

import org.jboss.metadata.spi.MetaData;
import org.jboss.metadata.spi.signature.ConstructorSignature;
import org.jboss.metadata.spi.signature.FieldSignature;
import org.jboss.metadata.spi.signature.MethodSignature;

/**
 * Decorator that understands MDR metadata for a type
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class MDRAnnotatedType<X> extends MDRAnnotated implements AnnotatedType<X>
{
   /** The decorated constructors */
   private final Set<AnnotatedConstructor<X>> annotatedConstructors;
   
   /** The decorated fields */
   private final Set<AnnotatedField<? super X>> annotatedFields;
   
   /** The decorated methods */
   private final Set<AnnotatedMethod<? super X>> annotatedMethods;

   /**
    * Constructor
    * 
    * @param delegate the delegate
    * @param metaData the MDR metadata 
    */
   MDRAnnotatedType(AnnotatedType<X> delegate, MetaData metaData)
   {
      super(delegate, metaData);
      annotatedConstructors = initialiseConstructors();
      annotatedFields = initialiseAnnotatedFields();
      annotatedMethods = initialiseMethods();
   }
   
   /**
    * Decorates the fields if MDR metadata is present
    * 
    * @return the fields 
    */
   private Set<AnnotatedField<? super X>> initialiseAnnotatedFields()
   {
      if (getMetaData() == null)
         return getDelegate().getFields();

      Set<AnnotatedField<? super X>> fields = new HashSet<AnnotatedField<? super X>>();
      for (AnnotatedField<? super X> field : getDelegate().getFields())
      {
         MetaData fieldMetaData = getMetaData().getComponentMetaData(new FieldSignature(field.getJavaMember()));
         fields.add(new MDRAnnotatedField<X>(this, field, fieldMetaData));
      }
      return fields;
   }
   
   /**
    * Decorates the constructors if MDR metadata is present
    * 
    * @return the constructors
    */
   private Set<AnnotatedConstructor<X>> initialiseConstructors()
   {
      if (getMetaData() == null)
         return getDelegate().getConstructors();

      Set<AnnotatedConstructor<X>> constructors = new HashSet<AnnotatedConstructor<X>>();
      for (AnnotatedConstructor<X> constructor : getDelegate().getConstructors())
      {
         MetaData constructorMetaData = getMetaData().getComponentMetaData(new ConstructorSignature(constructor.getJavaMember()));
         constructors.add(new MDRAnnotatedConstructor<X>(this, constructor, constructorMetaData, getMetaData()));
      }
      return constructors;
   }
   
   /**
    * Decorates the methods if MDR metadata is present
    * 
    * @return the methods
    */
   private Set<AnnotatedMethod<? super X>> initialiseMethods()
   {
      if (getMetaData() == null)
         return getDelegate().getMethods();

      Set<AnnotatedMethod<? super X>> methods = new HashSet<AnnotatedMethod<? super X>>();
      for (AnnotatedMethod<? super X> method : getDelegate().getMethods())
      {
         MetaData MethodMetaData = getMetaData().getComponentMetaData(new MethodSignature(method.getJavaMember()));
         methods.add(new MDRAnnotatedMethod<X>(this, method, MethodMetaData, getMetaData()));
      }
      return methods;
   }
   
   /**
    * Gets the delegate
    * 
    * @return the delegate
    */
   @Override
   @SuppressWarnings("unchecked")
   AnnotatedType<X> getDelegate()
   {
      return (AnnotatedType<X>)super.getDelegate();
   }
   
   /**
    * Gets the constructors decorated with MDR metadata
    * 
    * @return the constructors
    */
   public Set<AnnotatedConstructor<X>> getConstructors()
   {
      return annotatedConstructors;
   }

   /**
    * Gets the fields decorated with MDR metadata
    * 
    * @return the fields
    */
   public Set<AnnotatedField<? super X>> getFields()
   {
      return annotatedFields;
   }

   /**
    * Gets the methods decorated with MDR metadata
    * 
    * @return the methods
    */
   public Set<AnnotatedMethod<? super X>> getMethods()
   {
      return annotatedMethods;
   }

   /**
    * Gets the java class wrapped by this
    * 
    * @return the java class
    */
   public Class<X> getJavaClass()
   {
      return getDelegate().getJavaClass();
   }

   @Override
   public String toString()
   {
      return "MDRAnnotatedType{annotations=" + getAnnotations() + "; class " + getJavaClass().getName() + "}";
   }
}
