/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.weld.plugins.annotated;

import java.lang.annotation.Annotation;
import java.util.Set;

import javax.enterprise.inject.spi.Annotated;
import javax.enterprise.inject.spi.AnnotatedMember;
import javax.enterprise.inject.spi.AnnotatedType;
import javax.inject.Inject;

import org.jboss.kernel.weld.metadata.api.annotations.Weld;
import org.jboss.metadata.spi.MetaData;

/**
 * Decorator that understands MDR metadata for a field, constructor or method
 * 
 * @param <X> ?
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public abstract class MDRAnnotatedMember<X> extends MDRAnnotated implements AnnotatedMember<X>
{
   private MDRAnnotatedType<X> owner;
   private volatile AnnotatedType<X> declaringOwner;
   
   /**
    * Constructor
    *
    * @param owner the annotated type containing this field
    * @param delegate the delegate
    * @param metaData the MDR metadata for the method or constructor
    */
   public MDRAnnotatedMember(MDRAnnotatedType<X> owner, Annotated delegate, MetaData metaData)
   {
      super(delegate, metaData);
      if (owner == null)
         throw new IllegalArgumentException("Null owner");
      this.owner = owner;
   }

   /**
    * Gets the delegate
    * 
    * @return the delegate
    */
   @Override
   @SuppressWarnings("unchecked")
   AnnotatedMember<X> getDelegate()
   {
      return (AnnotatedMember<X>)super.getDelegate();
   }
   
   /**
    * Gets the type declaring this member
    * 
    * @return the decorated type with MDR metadata
    */
   public AnnotatedType<X> getDeclaringType()
   {
      AnnotatedType<X> declaring = getDelegate().getDeclaringType();
      if (declaring.getJavaClass() != owner.getJavaClass())
      {
         if (declaringOwner == null)
         {
            AnnotatedType<X> type = new MDRAnnotatedType<X>(declaring, owner.getMetaData());
            declaringOwner = type;
         }
         return declaringOwner;
      }
      return owner;
   }

   /**
    * Is member static
    * 
    * @return true if member is static
    */
   public boolean isStatic()
   {
      return getDelegate().isStatic();
   }

   protected Set<Annotation> trimInjectIfWeldAbsent(Set<Annotation> annotations)
   {
      if (!super.isAnnotationPresent(Weld.class) && super.isAnnotationPresent(Inject.class))
      {
         for (Annotation ann : annotations)
         {
            if (ann.annotationType() == Inject.class)
               annotations.remove(ann);
         }
      }
      return annotations;
   }

   public boolean isAnnotationPresent(Class<? extends Annotation> annotationType)
   {
      if (annotationType.equals(Inject.class) && !super.isAnnotationPresent(Weld.class))
         return false;

      return super.isAnnotationPresent(annotationType);
   }
   
   public <T extends Annotation> T getAnnotation(Class<T> annotationType)
   {
      if (annotationType.equals(Inject.class) && !isAnnotationPresent(Weld.class))
         return null;
      
      return super.getAnnotation(annotationType);
   }

}
