/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.logging;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;

/**
 * Logger wrapper that tries to dynamically load a log4j class to
 * determine if log4j is available in the VM. If it is the case,
 * a log4j delegate is built and used. In the contrary, a null
 * logger is used. This class cannot directly reference log4j
 * classes otherwise the JVM will try to load it and make it fail.
 * To set
 *
 * <p>Only exposes the relevent factory and logging methods.
 * 
 * <p>For JBoss the logging should be done as follows:
 * <ul>
 * <li>FATAL - JBoss is likely to/will crash
 * <li>ERROR - A definite problem
 * <li>WARN - Likely to be a problem, or it could be JBoss
 *            detected a problem it can recover from
 * <li>INFO - Lifecycle low volume, e.g. "Bound x into jndi",
 *            things that are of interest to a user
 * <li>DEBUG - Lifecycle low volume but necessarily of interest 
 *             to the user, e.g. "Starting listener thread"
 * <li>TRACE - High volume detailed logging
 * </ul>
 *
 * @see #isTraceEnabled
 * @see #trace(Object)
 * @see #trace(Object,Throwable)
 *
 * @version <tt>$Revision: 3403 $</tt>
 * @author  Scott.Stark@jboss.org
 * @author  <a href="mailto:jason@planet57.com">Jason Dillon</a>
 * @author  <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>
 */
public class Logger implements Serializable
{
   private static volatile LoggerPlugin plugin;

    /**
    * Levels used by this logging API.
    */
   public enum Level {
       FATAL,
       ERROR,
       WARN,
       INFO,
       DEBUG,
       TRACE,
   }

   /** Serialization */
   private static final long serialVersionUID = 4232175575988879434L;

   private static final String FQCN = Logger.class.getName();

   /** The system property to look for an externalized LoggerPlugin implementation class */
   protected static String PLUGIN_CLASS_PROP = "org.jboss.logging.Logger.pluginClass";

   /** The default LoggerPlugin implementation is log4j */
   protected static final String LOG4J_PLUGIN_CLASS_NAME = "org.jboss.logging.log4j.Log4jLoggerPlugin";

   /** The LoggerPlugin implementation class to use */
   protected static Class<?> pluginClass = null;

   /** The class name of the LoggerPlugin implementation class to use */
   protected static String pluginClassName = null;

   static
   {
      init();
   }

   /** The logger name. */
   private final String name;

   /** The resource bundle name. */
   private final String resourceBundleName;

   /** The logger plugin delegate */
   protected transient LoggerPluginInstance loggerDelegate = null;

   /**
    * The LoggerPlugin implementation class name in use
    * 
    * @return LoggerPlugin implementation class name
    */
   public static String getPluginClassName()
   {
      return Logger.pluginClassName;
   }

   /**
    * Set the LoggerPlugin implementation class name in use
    * 
    * @param pluginClassName the LoggerPlugin implementation class name
    */
   public static void setPluginClassName(String pluginClassName)
   {
      if (! pluginClassName.equals(Logger.pluginClassName))
      {
         Logger.pluginClassName = pluginClassName;
         init();
      }
   }

   /**
    * Creates new Logger the given logger name.
    *
    * @param name the logger name.
    */
   Logger(final String name, final String resourceBundleName, final LoggerPluginInstance loggerDelegate)
   {
      this.name = name;
      this.resourceBundleName = resourceBundleName;
      this.loggerDelegate = loggerDelegate;
   }

   /**
    * Return the name of this logger.
    *
    * @return The name of this logger.
    */
   public String getName()
   {
      return name;
   }

   /**
    * Get the logger plugin delegate
    * 
    * @return the delegate
    */
   public LoggerPluginInstance getLoggerPluginInstance()
   {
      return loggerDelegate;
   }

   /**
    * Check to see if the TRACE level is enabled for this logger.
    *
    * @return true if a {@link #trace(Object)} method invocation would pass
    *         the msg to the configured appenders, false otherwise.
    */
   public boolean isTraceEnabled()
   {
      return loggerDelegate.isEnabled(Level.TRACE);
   }

   /**
    * Issue a log msg with a level of TRACE.
    * 
    * @param message the message
    */
   public void trace(Object message)
   {
      loggerDelegate.log(Level.TRACE, FQCN, message, null, null);
   }

   /**
    * Issue a log msg and throwable with a level of TRACE.
    * 
    * @param message the message
    * @param t the throwable
    */
   public void trace(Object message, Throwable t)
   {
       loggerDelegate.log(Level.TRACE, FQCN, message, null, t);
   }

   /**
    * Issue a log msg and throwable with a level of TRACE and a specific logger class name.
    *
    * @param loggerFqcn the logger class name
    * @param message the message
    * @param t the throwable
    */
   public void trace(String loggerFqcn, Object message, Throwable t)
   {
      loggerDelegate.log(Level.TRACE, loggerFqcn, message, null, t);
   }

    /**
     * Issue a log message with parameters with a level of TRACE.
     *
     * @param message the message
     * @param params the message parameters
     */
   public void trace(Object message, Object[] params)
   {
      loggerDelegate.log(Level.TRACE, FQCN, message, params, null);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of TRACE.
     *
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void trace(Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.TRACE, FQCN, message, params, t);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of TRACE.
     *
     * @param loggerFqcn the logger class name
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void trace(String loggerFqcn, Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.TRACE, loggerFqcn, message, params, t);
   }

   /**
    * Issue a formatted log message with a level of TRACE.
    *
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the parameters
    */
   public void tracef(String format, Object... params)
   {
      loggerDelegate.logf(Level.TRACE, FQCN, format, params, null);
   }

   /**
    * Issue a formatted log message with a level of TRACE.
    *
    * @param t the throwable
    * @param format the format string, as per {@link String#format(String, Object[])}
    * @param params the parameters
    */
   public void tracef(Throwable t, String format, Object... params)
   {
      loggerDelegate.logf(Level.TRACE, FQCN, format, params, t);
   }

   /**
    * Check to see if the DEBUG level is enabled for this logger.
    *
    * @return true if a {@link #debug(Object)} method invocation would pass
    *         the msg to the configured appenders, false otherwise.
    */
   public boolean isDebugEnabled()
   {
      return loggerDelegate.isEnabled(Level.DEBUG);
   }

   /**
    * Issue a log message with a level of DEBUG.
    *
    * @param message the message
    */
   public void debug(Object message)
   {
      loggerDelegate.log(Level.DEBUG, FQCN, message, null, null);
   }

   /**
    * Issue a log message and throwable with a level of DEBUG.
    *
    * @param message the message
    * @param t the throwable
    */
   public void debug(Object message, Throwable t)
   {
       loggerDelegate.log(Level.DEBUG, FQCN, message, null, t);
   }

   /**
    * Issue a log message and throwable with a level of DEBUG and a specific logger class name.
    *
    * @param loggerFqcn the logger class name
    * @param message the message
    * @param t the throwable
    */
   public void debug(String loggerFqcn, Object message, Throwable t)
   {
      loggerDelegate.log(Level.DEBUG, loggerFqcn, message, null, t);
   }

    /**
     * Issue a log message with parameters with a level of DEBUG.
     *
     * @param message the message
     * @param params the message parameters
     */
   public void debug(Object message, Object[] params)
   {
      loggerDelegate.log(Level.DEBUG, FQCN, message, params, null);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of DEBUG.
     *
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void debug(Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.DEBUG, FQCN, message, params, t);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of DEBUG.
     *
     * @param loggerFqcn the logger class name
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void debug(String loggerFqcn, Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.DEBUG, loggerFqcn, message, params, t);
   }

   /**
    * Issue a formatted log message with a level of DEBUG.
    *
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the parameters
    */
   public void debugf(String format, Object... params)
   {
      loggerDelegate.logf(Level.DEBUG, FQCN, format, params, null);
   }

   /**
    * Issue a formatted log message with a level of DEBUG.
    *
    * @param t the throwable
    * @param format the format string, as per {@link String#format(String, Object[])}
    * @param params the parameters
    */
   public void debugf(Throwable t, String format, Object... params)
   {
      loggerDelegate.logf(Level.DEBUG, FQCN, format, params, t);
   }

   /**
    * Check to see if the INFO level is enabled for this logger.
    *
    * @return true if a {@link #info(Object)} method invocation would pass
    *         the msg to the configured appenders, false otherwise.
    */
   public boolean isInfoEnabled()
   {
      return loggerDelegate.isEnabled(Level.INFO);
   }

   /**
    * Issue a log message with a level of INFO.
    *
    * @param message the message
    */
   public void info(Object message)
   {
      loggerDelegate.log(Level.INFO, FQCN, message, null, null);
   }

   /**
    * Issue a log message and throwable with a level of INFO.
    *
    * @param message the message
    * @param t the throwable
    */
   public void info(Object message, Throwable t)
   {
       loggerDelegate.log(Level.INFO, FQCN, message, null, t);
   }

   /**
    * Issue a log message and throwable with a level of INFO and a specific logger class name.
    *
    * @param loggerFqcn the logger class name
    * @param message the message
    * @param t the throwable
    */
   public void info(String loggerFqcn, Object message, Throwable t)
   {
      loggerDelegate.log(Level.INFO, loggerFqcn, message, null, t);
   }

    /**
     * Issue a log message with parameters with a level of INFO.
     *
     * @param message the message
     * @param params the message parameters
     */
   public void info(Object message, Object[] params)
   {
      loggerDelegate.log(Level.INFO, FQCN, message, params, null);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of INFO.
     *
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void info(Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.INFO, FQCN, message, params, t);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of INFO.
     *
     * @param loggerFqcn the logger class name
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void info(String loggerFqcn, Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.INFO, loggerFqcn, message, params, t);
   }

   /**
    * Issue a formatted log message with a level of INFO.
    *
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the parameters
    */
   public void infof(String format, Object... params)
   {
      loggerDelegate.logf(Level.INFO, FQCN, format, params, null);
   }

   /**
    * Issue a formatted log message with a level of INFO.
    *
    * @param t the throwable
    * @param format the format string, as per {@link String#format(String, Object[])}
    * @param params the parameters
    */
   public void infof(Throwable t, String format, Object... params)
   {
      loggerDelegate.logf(Level.INFO, FQCN, format, params, t);
   }

   /**
    * Issue a log message with a level of WARN.
    *
    * @param message the message
    */
   public void warn(Object message)
   {
      loggerDelegate.log(Level.WARN, FQCN, message, null, null);
   }

   /**
    * Issue a log message and throwable with a level of WARN.
    *
    * @param message the message
    * @param t the throwable
    */
   public void warn(Object message, Throwable t)
   {
       loggerDelegate.log(Level.WARN, FQCN, message, null, t);
   }

   /**
    * Issue a log message and throwable with a level of WARN and a specific logger class name.
    *
    * @param loggerFqcn the logger class name
    * @param message the message
    * @param t the throwable
    */
   public void warn(String loggerFqcn, Object message, Throwable t)
   {
      loggerDelegate.log(Level.WARN, loggerFqcn, message, null, t);
   }

    /**
     * Issue a log message with parameters with a level of WARN.
     *
     * @param message the message
     * @param params the message parameters
     */
   public void warn(Object message, Object[] params)
   {
      loggerDelegate.log(Level.WARN, FQCN, message, params, null);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of WARN.
     *
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void warn(Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.WARN, FQCN, message, params, t);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of WARN.
     *
     * @param loggerFqcn the logger class name
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void warn(String loggerFqcn, Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.WARN, loggerFqcn, message, params, t);
   }

   /**
    * Issue a formatted log message with a level of WARN.
    *
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the parameters
    */
   public void warnf(String format, Object... params)
   {
      loggerDelegate.logf(Level.WARN, FQCN, format, params, null);
   }

   /**
    * Issue a formatted log message with a level of WARN.
    *
    * @param t the throwable
    * @param format the format string, as per {@link String#format(String, Object[])}
    * @param params the parameters
    */
   public void warnf(Throwable t, String format, Object... params)
   {
      loggerDelegate.logf(Level.WARN, FQCN, format, params, t);
   }

   /**
    * Issue a log message with a level of ERROR.
    *
    * @param message the message
    */
   public void error(Object message)
   {
      loggerDelegate.log(Level.ERROR, FQCN, message, null, null);
   }

   /**
    * Issue a log message and throwable with a level of ERROR.
    *
    * @param message the message
    * @param t the throwable
    */
   public void error(Object message, Throwable t)
   {
       loggerDelegate.log(Level.ERROR, FQCN, message, null, t);
   }

   /**
    * Issue a log message and throwable with a level of ERROR and a specific logger class name.
    *
    * @param loggerFqcn the logger class name
    * @param message the message
    * @param t the throwable
    */
   public void error(String loggerFqcn, Object message, Throwable t)
   {
      loggerDelegate.log(Level.ERROR, loggerFqcn, message, null, t);
   }

    /**
     * Issue a log message with parameters with a level of ERROR.
     *
     * @param message the message
     * @param params the message parameters
     */
   public void error(Object message, Object[] params)
   {
      loggerDelegate.log(Level.ERROR, FQCN, message, params, null);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of ERROR.
     *
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void error(Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.ERROR, FQCN, message, params, t);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of ERROR.
     *
     * @param loggerFqcn the logger class name
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void error(String loggerFqcn, Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.ERROR, loggerFqcn, message, params, t);
   }

   /**
    * Issue a formatted log message with a level of ERROR.
    *
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the parameters
    */
   public void errorf(String format, Object... params)
   {
      loggerDelegate.logf(Level.ERROR, FQCN, format, params, null);
   }

   /**
    * Issue a formatted log message with a level of ERROR.
    *
    * @param t the throwable
    * @param format the format string, as per {@link String#format(String, Object[])}
    * @param params the parameters
    */
   public void errorf(Throwable t, String format, Object... params)
   {
      loggerDelegate.logf(Level.ERROR, FQCN, format, params, t);
   }

   /**
    * Issue a log message with a level of FATAL.
    *
    * @param message the message
    */
   public void fatal(Object message)
   {
      loggerDelegate.log(Level.FATAL, FQCN, message, null, null);
   }

   /**
    * Issue a log message and throwable with a level of FATAL.
    *
    * @param message the message
    * @param t the throwable
    */
   public void fatal(Object message, Throwable t)
   {
       loggerDelegate.log(Level.FATAL, FQCN, message, null, t);
   }

   /**
    * Issue a log message and throwable with a level of FATAL and a specific logger class name.
    *
    * @param loggerFqcn the logger class name
    * @param message the message
    * @param t the throwable
    */
   public void fatal(String loggerFqcn, Object message, Throwable t)
   {
      loggerDelegate.log(Level.FATAL, loggerFqcn, message, null, t);
   }

    /**
     * Issue a log message with parameters with a level of FATAL.
     *
     * @param message the message
     * @param params the message parameters
     */
   public void fatal(Object message, Object[] params)
   {
      loggerDelegate.log(Level.FATAL, FQCN, message, params, null);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of FATAL.
     *
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void fatal(Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.FATAL, FQCN, message, params, t);
   }

    /**
     * Issue a log message with parameters and a throwable with a level of FATAL.
     *
     * @param loggerFqcn the logger class name
     * @param message the message
     * @param params the message parameters
     * @param t the throwable
     */
   public void fatal(String loggerFqcn, Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(Level.FATAL, loggerFqcn, message, params, t);
   }

   /**
    * Issue a formatted log message with a level of FATAL.
    *
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the parameters
    */
   public void fatalf(String format, Object... params)
   {
      loggerDelegate.logf(Level.FATAL, FQCN, format, params, null);
   }

   /**
    * Issue a formatted log message with a level of FATAL.
    *
    * @param t the throwable
    * @param format the format string, as per {@link String#format(String, Object[])}
    * @param params the parameters
    */
   public void fatalf(Throwable t, String format, Object... params)
   {
      loggerDelegate.logf(Level.FATAL, FQCN, format, params, t);
   }

   /**
    * Log a message at the given level.
    *
    * @param level the level
    * @param message the message
    */
   public void log(Level level, Object message)
   {
      loggerDelegate.log(level, FQCN, message, null, null);
   }

   /**
    * Log a message at the given level.
    *
    * @param level the level
    * @param message the message
    * @param params the message parameters
    */
   public void log(Level level, Object message, Object[] params)
   {
      loggerDelegate.log(level, FQCN, message, params, null);
   }

   /**
    * Log a message at the given level.
    *
    * @param level the level
    * @param message the message
    * @param params the message parameters
    * @param t the throwable cause
    */
   public void log(Level level, Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(level, FQCN, message, params, t);
   }

   /**
    * Log a message at the given level.
    *
    * @param loggerFqcn the logger class name
    * @param level the level
    * @param message the message
    * @param params the message parameters
    * @param t the throwable cause
    */
   public void log(String loggerFqcn, Level level, Object message, Object[] params, Throwable t)
   {
      loggerDelegate.log(level, loggerFqcn, message, params, t);
   }

   /**
    * Log a message at the given level.
    *
    * @param level the level
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the message parameters
    */
   public void logf(Level level, String format, Object... params)
   {
      loggerDelegate.logf(level, FQCN, format, params, null);
   }

   /**
    * Log a message at the given level.
    *
    * @param level the level
    * @param t the throwable cause
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the message parameters
    */
   public void logf(Level level, Throwable t, String format, Object... params)
   {
      loggerDelegate.logf(level, FQCN, format, params, t);
   }

   /**
    * Log a message at the given level.
    *
    * @param loggerFqcn the logger class name
    * @param level the level
    * @param t the throwable cause
    * @param format the format string as per {@link String#format(String, Object[])} or resource bundle key therefor
    * @param params the message parameters
    */
   public void logf(String loggerFqcn, Level level, Throwable t, String format, Object... params)
   {
      loggerDelegate.logf(level, loggerFqcn, format, params, t);
   }

   /**
    * Custom serialization to reinitalize the delegate
    * 
    * @param stream the object stream
    * @throws IOException for any error
    * @throws ClassNotFoundException if a class is not found during deserialization
    */
   private void readObject(ObjectInputStream stream) throws IOException, ClassNotFoundException
   {
      // restore non-transient fields (aka name)
      stream.defaultReadObject();

      // Restore logging
      if (pluginClass == null)
      {
         init();
      }
   }

   private Object readResolve()
   {
      return getDelegatePluginInstance(name, resourceBundleName).getLogger();
   }

   /**
    * Get a logger instance with the given name using the given resource bundle (if supported by the underlying
    * framework).
    *
    * @param name the logger category name
    * @param resourceBundleName the resource bundle name
    * @return the logger
    */
   public static Logger getI18nLogger(String name, String resourceBundleName)
   {
      return getDelegatePluginInstance(name, resourceBundleName).getLogger();
   }

   /**
    * Get a logger instance with the given name using the given resource bundle (if supported by the underlying
    * framework).
    *
    * @param clazz the class whose name will be used as the logger category name
    * @param resourceBundleName the resource bundle name
    * @return the logger
    */
   public static Logger getI18nLogger(Class<?> clazz, String resourceBundleName)
   {
      return getI18nLogger(clazz.getName(), resourceBundleName);
   }

   /**
    * Get a Logger instance given the logger name.
    *
    * @param name the logger name
    * @return the logger
    */
   public static Logger getLogger(String name)
   {
      return getI18nLogger(name, null);
   }

   /**
    * Get a Logger instance given the logger name with the given suffix.
    *
    * <p>This will include a logger seperator between classname and suffix
    *
    * @param name the logger name
    * @param suffix a suffix to append to the classname.
    * @return the logger
    */
   public static Logger getLogger(String name, String suffix)
   {
      return getLogger(name == null || name.length() == 0 ? suffix : name + "." + suffix);
   }

   /**
    * Get a Logger instance given the name of a class. This simply
    * calls create(clazz.getName()).
    *
    * @param clazz the Class whose name will be used as the logger name
    * @return the logger
    */
   public static Logger getLogger(Class<?> clazz)
   {
      return getLogger(clazz.getName());
   }

   /**
    * Get a Logger instance given the name of a class with the given suffix.
    *
    * <p>This will include a logger seperator between classname and suffix
    *
    * @param clazz the Class whose name will be used as the logger name.
    * @param suffix a suffix to append to the classname.
    * @return the logger
    */
   public static Logger getLogger(Class<?> clazz, String suffix)
   {
      return getLogger(clazz.getName(), suffix);
   }

   /**
    * Get the logger plugin.
    *
    * @return the logger plugin
    */
   public LoggerPlugin getLoggerPlugin()
   {
      return getLoggerPluginInstance().getLoggerPlugin();
   }

   /**
    * Get the delegate plugin
    * 
    * @param name the name of the logger
    * @param resourceBundleName the default resource bundle name to use
    * @return the plugin
    */
   static LoggerPluginInstance getDelegatePluginInstance(String name, String resourceBundleName)
   {
      LoggerPlugin plugin;
      LoggerPluginInstance instance;
      try
      {
         plugin = (LoggerPlugin) pluginClass.newInstance();
      }
      catch (Throwable e)
      {
         plugin = new NullLoggerPlugin();
      }
      try
      {
         instance = plugin.getInstance(name, resourceBundleName);
      }
      catch (Throwable e)
      {
         System.err.println("Failed to initalize plugin: " + plugin + ", cause: " + e.toString());
         instance = new NullLoggerPluginInstance(name, NullLoggerPlugin.INSTANCE);
      }

      return instance;
   }

   /**
    * Initialize the LoggerPlugin class to use as the delegate to the
    * logging system. This first checks to see if a pluginClassName has
    * been specified via the {@link #setPluginClassName(String)} method,
    * then the PLUGIN_CLASS_PROP system property and finally the
    * LOG4J_PLUGIN_CLASS_NAME default. If the LoggerPlugin implementation
    * class cannot be loaded the default NullLoggerPlugin will be used.
    */
   protected static void init()
   {
      try
      {
         // See if there is a PLUGIN_CLASS_PROP specified
         if (pluginClassName == null)
         {
            pluginClassName = System.getProperty(PLUGIN_CLASS_PROP, LOG4J_PLUGIN_CLASS_NAME);
         }

         // Try to load the plugin via the TCL
         ClassLoader cl = Thread.currentThread().getContextClassLoader();
         pluginClass = cl.loadClass(pluginClassName);
      }
      catch (Throwable e)
      {
         // The plugin could not be setup, default to a null logger
         pluginClass = NullLoggerPluginInstance.class;
      }
   }
}
