/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.logging.metadata;

import org.jboss.util.collection.ConcurrentReferenceHashMap;
import org.jboss.logmanager.LogContext;
import static org.jboss.util.collection.ConcurrentReferenceHashMap.ReferenceType.*;
import java.util.concurrent.ConcurrentMap;

/**
 * A named registry for log contexts.  It retains only weak references to the log context instance.
 */
public final class LogContextRegistry {

    private static final LogContextRegistry INSTANCE = new LogContextRegistry();

    private final ConcurrentMap<String, LogContext> map = new ConcurrentReferenceHashMap<String, LogContext>(STRONG, WEAK);

    private LogContextRegistry() {
        map.put("system", LogContext.getSystemLogContext());
    }

    /**
     * Get the named log context.
     *
     * @param name the context name
     * @return the context
     * @throws IllegalArgumentException if no such log context could be found
     */
    public LogContext get(String name) throws IllegalArgumentException {
        final LogContext context = map.get(name);
        if (context == null) {
            throw new IllegalArgumentException("No such log context '" + name + "'");
        }
        return context;
    }

    /**
     * Get the named log context, creating it if it does not exist.
     *
     * @param name the context name
     * @return the context
     */
    public LogContext getOrCreate(String name) {
        final LogContext context = map.get(name);
        if (context == null) {
            final LogContext created = LogContext.create();
            final LogContext appearing = map.putIfAbsent(name, created);
            return appearing == null ? created : appearing;
        } else {
            return context;
        }
    }

    /**
     * Get the single instance.
     *
     * @return the single instance
     */
    public static LogContextRegistry getInstance() {
        return INSTANCE;
    }
}
