/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.api.factory;

import org.jboss.managed.spi.factory.ManagedObjectBuilder;
import org.jboss.util.JBossStringBuilder;

/**
 * Configurable managed object definition.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class ConfigurableManagedObjectDefinition extends BaseManagedObjectDefinition
{
   private ManagedObjectFactory factory;

   public ConfigurableManagedObjectDefinition()
   {
   }

   public ConfigurableManagedObjectDefinition(Class<?> type, ManagedObjectBuilder builder)
   {
      super(type, builder);
   }

   public ConfigurableManagedObjectDefinition(ManagedObjectFactory factory)
   {
      this(factory, null, null);
   }

   public ConfigurableManagedObjectDefinition(ManagedObjectFactory factory, Class<?> type, ManagedObjectBuilder builder)
   {
      super(type, builder);
      this.factory = factory;
   }

   /**
    * Get factory internal.
    *
    * This method should be used
    * when internally accessing MO factory
    * as it will return default instance in case
    * factory is not explicitly set. 
    *
    * @return the MO factory
    */
   protected ManagedObjectFactory getMOFactory()
   {
      if (factory == null)
         factory = ManagedObjectFactory.getInstance();

      return factory;     
   }

   /**
    * Add builder to factory.
    */
   public void start()
   {
      if (isValid() == false)
         throw new IllegalArgumentException("Definition is invalid: " + toString());

      getMOFactory().setBuilder(getType(), getBuilder());
   }

   /**
    * Remove builder from factory.
    */
   public void stop()
   {
      if (factory == null)
         throw new IllegalArgumentException("Missing managed object factory");
      if (getType() == null)
         throw new IllegalArgumentException("Missing type");

      getMOFactory().setBuilder(getType(), null);
   }

   /**
    * Get managed obejct factory.
    *
    * @return the managed object factory
    */
   public ManagedObjectFactory getFactory()
   {
      return factory;
   }

   /**
    * Set managed object factory.
    *
    * @param factory the managed object factory
    */
   public void setFactory(ManagedObjectFactory factory)
   {
      this.factory = factory;
   }

   @Override
   public void toShortString(JBossStringBuilder buffer)
   {
      super.toShortString(buffer);
      buffer.append(", factory=").append(factory);
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      super.toString(buffer);
      buffer.append(", factory=").append(factory);
   }
}