package org.jboss.maven.plugins.thirdparty;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.FileUtils;
import org.jboss.maven.plugins.thirdparty.util.JarUtil;

/**
 * Maven goal for generating a thirdparty directory structure using artifact downloaded
 * from a maven repository.
 * 
 * @phase generate-resources
 * @goal build-thirdparty
 * @requiresDependencyResolution test
 * @since 2.0
 * 
 * @author Paul Gier
 * @version $Revision: 198 $
 */
public class BuildThirdpartyMojo extends AbstractMojo
{

   private static final String[] EMPTY_STRING_ARRAY = {};

   private static final String[] DEFAULT_INCLUDES = {"**/**"};

   /**
    * The Maven Project Object
    *
    * @parameter expression="${project}"
    * @readonly
    */
   protected MavenProject project;

   /**
    * This parameter provides an option to skip execution of the goal. 
    * 
    * @parameter default-value=false
    */
   private boolean skip;

   /**
    * The directory to which the dependency jars should be written.
    * 
    * @parameter default-value="${project.build.directory}"
    */
   private File outputDirectory;
   
   /**
    * This folder contains thirdparty resources.  The contents of the folder
    * are just copied to the outputDirectory.
    * 
    * @parameter default-value="src/main/resources"
    * @deprecated Resources should be packaged up in a "resources" artifact.
    */
   private File thirdpartyResources;
   
   /**
    * File patterns to include in the thirdparty resources directory.
    * 
    * @parameter
    * @deprecated
    */
   private Set resourceIncludes;
   
   /**
    * File patterns to exclude from the resources directory.
    * 
    * @parameter
    * @deprecated
    */
   private Set resourceExcludes;
   
   /**
    * Determines whether to copy resource files from the thirdpartyResources directory.
    * 
    * @parameter default-value="false"
    */
   private boolean copyResources;
   
   /**
    * Determines whether to look for source jars in the local repository and
    * copy them to the thirdparty directory if they are found.
    * 
    * @parameter default-value="false"
    */
   private boolean copySources;
   
   /**
    * A map of the maven dependencies to the appropriate thirdparty name.
    * 
    * @parameter
    */
   private Set mappedDependencies;
   
   /**
    * A list of the dependency scopes to include in the download.
    * By default, only dependencies the compile scope is included.
    * 
    * @parameter
    */
   private List includedScopes;
   
   /**
    * The file to write the libraries.ent information.
    * 
    * @parameter default-value="${project.build.directory}/libraries.ent"
    */
   private File librariesEnt;
   
   /**
    * An additional file to append at the end of the libraries.ent file.
    * 
    * @parameter
    */
   private File aliasesEnt;
   
   /**
    * Translations that use a wildcard (*) for the artifactId.
    */
   private Map groupIdTranslations = new HashMap();
   
   /**
    * Main execution path of the plugin.  Copies downloaded dependencies to thirdparty directory 
    * and generates component-info.xml files.
    */
   public void execute() throws MojoExecutionException
   {
      this.getLog().debug("Executing build thirdparty mojo");
      
      if ( skip )
      {
         getLog().info( "Skipping execution." );
         return;
      }
      if ( includedScopes == null || includedScopes.size() == 0 )
      {
         includedScopes = new ArrayList();
         includedScopes.add( "compile" );
      }

      // Copy the dependency mappings into a map object for easy lookup.
      Map tpDepNameMap = new HashMap();
      Iterator depIter = mappedDependencies.iterator();
      while ( depIter.hasNext() )
      {
         Dependency dep = (Dependency)depIter.next();
         tpDepNameMap.put( dep.getResolutionId(), dep);
         // Add wildcard groupId translations
         if ( dep.getArtifactId().equals( "*" ) )
         {
            groupIdTranslations.put( dep.getGroupId(), dep.getComponentId() );
         }
      }
      
      // Copy the dependencies and generate the component-info objects.
      Map compInfoMap = copyDependenciesAndGenrateCompInfo(tpDepNameMap);
      
      // Write comp-info files
      getLog().info( "Generating component info files..." );
      try 
      {
         ComponentInfoWriter.loadTemplate();
      }
      catch ( IOException e )
      {
         throw new MojoExecutionException( "Unable to load componentInfo template.", e );
      }
      Iterator componentsIter = compInfoMap.keySet().iterator();
      while ( componentsIter.hasNext() )
      {
         String componentId = (String)componentsIter.next();
         ComponentInfo compInfo = (ComponentInfo)compInfoMap.get( componentId );
         File compInfoDir = new File ( outputDirectory, componentId);
         File compInfoFile = new File ( compInfoDir, "component-info.xml" );
         try 
         {
            ComponentInfoWriter.writeComponentInfo( compInfo, compInfoFile );
         }
         catch ( IOException e )
         {
            getLog().warn( "Unable to write comp info file: " + compInfoFile );
            getLog().warn( e );
         }
      }

      // Copy resources folder
      if ( copyResources )
      {
         try 
         {
            copyResources( thirdpartyResources, outputDirectory );
         }
         catch ( IOException e )
         {
            getLog().warn( "Unable to copy resources folder: " + thirdpartyResources );
            getLog().warn( e );
         }
      }
      
      // Write the libraries.ent file
      try 
      {
         generateLibrariesEnt( compInfoMap );
      }
      catch ( IOException e )
      {
         getLog().warn( "Problem writing libraries ent file: " + librariesEnt );
         getLog().warn( e );
      }
      
   }

   /**
    * Copies the downloaded dependencies to the thirdparty directory and generates
    * component-info objects.
    * 
    * @param tpDepNameMap
    * @return Map of the componentId and the componentInfoObjects.
    * @throws MojoExecutionException
    */
   private Map copyDependenciesAndGenrateCompInfo(Map tpDepNameMap)
         throws MojoExecutionException
   {
      getLog().info( "Copying dependencies to thirdparty directories..." );
      Map compInfoMap = new HashMap();
      Set artifacts = project.getArtifacts();
      Iterator iter = artifacts.iterator();
      while ( iter.hasNext() )
      {
         Artifact artifact = (Artifact)iter.next();
         if ( artifact.getScope() == null )
         {
            artifact.setScope( "compile" );
         }
         if ( ! includedScopes.contains( artifact.getScope() ) )
         {
            getLog().debug( "artifact not included: " + artifact );
            continue;
         }
         
         File mavenArtifactFile = artifact.getFile();
         
         // Get's the ID of the artifact without the version
         String artifactResId = getDependencyResolutionId( artifact );
         
         Dependency tpdependency = (Dependency)tpDepNameMap.get( artifactResId );
        
         if ( tpdependency == null )
         {
            tpdependency = new Dependency( artifact );
            if ( groupIdTranslations.containsKey( artifact.getGroupId()) )
            {
               tpdependency.setComponentId( (String)groupIdTranslations.get( artifact.getGroupId() ) );
            }
         }
         else
         {
            if ( tpdependency.getVersion() == null )
            {
               tpdependency.setVersion( artifact.getVersion() );
            }
         }
         
         String componentId = tpdependency.getComponentId();
         File componentLibDir = new File( outputDirectory, tpdependency.getComponentId() + "/lib/" );
         componentLibDir.mkdirs();

         // If it's a resources file, we want to extract the resources instead of copying the jar.
         if ( artifact.getArtifactId().equals("resources"))
         {
            File componentDir = new File( outputDirectory, tpdependency.getComponentId() );
            try 
            {
               JarUtil.extractJarFile( artifact.getFile(), componentDir );
            }
            catch ( IOException e )
            {
               getLog().warn( "Unable to extract resources artifact: " + artifact );
               getLog().warn( e );
            }
            continue;
         }
         
         // Get the artifactId for the thirdparty jar
         String fileExt = getFileExtension( mavenArtifactFile );
         String compInfoArtifactId = tpdependency.getCompArtifactId() + "." + fileExt;
         File thirdpartyJarFile = new File ( componentLibDir, compInfoArtifactId );
         
         // Set up component-info
         ComponentInfo compInfo = (ComponentInfo)compInfoMap.get( componentId );
         if ( compInfo == null ) 
         {
            compInfo = new ComponentInfo();
            compInfo.setComponentId( componentId );
            compInfo.setVersion( tpdependency.getComponentVersion() );
         }

         compInfo.addArtifactId( compInfoArtifactId );
         
         if ( tpdependency.isExportArtifact() )
         {
            compInfo.addExport( compInfoArtifactId );
         }
         compInfoMap.put( componentId, compInfo );
         
         try 
         {
            getLog().debug( "Copying: " + mavenArtifactFile );
            getLog().debug( "to: " + thirdpartyJarFile );
            
            if ( mavenArtifactFile.length() != thirdpartyJarFile.length() )
            {
               FileUtils.copyFile( mavenArtifactFile, thirdpartyJarFile );
            }

            if ( this.copySources )
            {
               String sourceJarName = mavenArtifactFile.getName().replace(".jar", "-sources.jar");
               File sourceJarFile = new File ( mavenArtifactFile.getParentFile(), sourceJarName );
               if ( sourceJarFile.exists() )
               {
                  String tpSourceArtifactFileName = tpdependency.getCompArtifactId() + "-sources.jar";
                  File thirdpartySourceJar = new File( componentLibDir, tpSourceArtifactFileName );
                  if ( sourceJarFile.length() != thirdpartySourceJar.length() )
                  {
                     FileUtils.copyFile( sourceJarFile, thirdpartySourceJar );
                  }
               }
            }

         }
         catch (IOException e) 
         {
            throw new MojoExecutionException(" Unable to copy artifact: " + e, e);
         }
      }
      return compInfoMap;
   }

   public void copyResources(File resourceDirectory, File outputDirectory) throws IOException
   {

      if (!resourceDirectory.exists())
      {
         getLog().debug("Resource directory does not exist: " + resourceDirectory);
         return;
      }

      DirectoryScanner scanner = new DirectoryScanner();

      scanner.setBasedir(resourceDirectory);
      if ( resourceIncludes == null || resourceIncludes.isEmpty() )
      {
         scanner.setIncludes(DEFAULT_INCLUDES);
      }
      else
      {
         scanner.setIncludes((String[]) resourceIncludes.toArray(EMPTY_STRING_ARRAY));
      }

      if ( resourceExcludes != null && !resourceExcludes.isEmpty() )
      {
         scanner.setExcludes((String[]) resourceExcludes.toArray(EMPTY_STRING_ARRAY));
      }

      scanner.addDefaultExcludes();
      scanner.scan();

      List includedFiles = Arrays.asList(scanner.getIncludedFiles());

      for (Iterator j = includedFiles.iterator(); j.hasNext();)
      {
         String name = (String) j.next();

         String destination = name;

         File source = new File(resourceDirectory, name);

         File destinationFile = new File(outputDirectory, destination);

         if (!destinationFile.getParentFile().exists())
         {
            destinationFile.getParentFile().mkdirs();
         }

         FileUtils.copyFileIfModified(source, destinationFile);
      }
   }
   
   public void generateLibrariesEnt( Map compInfoMap ) throws IOException
   {
      getLog().info( "Creating " + librariesEnt.getName() + "..." );
      FileWriter fw = new FileWriter( librariesEnt );
      Set compInfoIds = compInfoMap.keySet();
      List compInfoIdList = new ArrayList();
      compInfoIdList.addAll(compInfoIds);
      Collections.sort( compInfoIdList );
      
      for ( int i=0; i<compInfoIdList.size(); ++i )
      {
         ComponentInfo compInfo = (ComponentInfo)compInfoMap.get( compInfoIdList.get(i) );
         fw.write( ComponentInfoWriter.getLibrariesEntEntry( compInfo ) );
      }
      
      if ( aliasesEnt != null && aliasesEnt.isFile() )
      {
         BufferedReader aliasesReader = new BufferedReader( new FileReader( aliasesEnt ) );
         String line = "";
         while ( ( line = aliasesReader.readLine() ) != null ) 
         {
            fw.write( line + "\n" );
         }
         aliasesReader.close();
      }
      
      fw.close();
      
   }
   
   /**
    * Returns the extension of a file name.
    * Does not include the "."
    */
   private String getFileExtension( File file )
   {
      String fileName = file.getName();
      int lastDot = fileName.lastIndexOf( '.' );
      
      if ( lastDot >= 0 )
      {
          return fileName.substring( lastDot + 1 );
      }
      else
      {
          return "";
      }
   }
      
   /**
    * Get an id that can be used when resolving the artifact
    * 
    * @param artifact
    * @return combination of the groupId, artifactId, and optionally the classifier.
    */
   private String getDependencyResolutionId( Artifact artifact )
   {
      StringBuffer id = new StringBuffer();
      id.append( artifact.getGroupId() );
      id.append( ":" );
      id.append( artifact.getArtifactId() );
      if ( artifact.getClassifier() != null )
      {
         id.append( ":" + artifact.getClassifier());
      }
      return id.toString();
   }
      
}
