/*
 * JBoss, the OpenSource J2EE webOS
 * 
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.maven.plugins.thirdparty;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * 
 * Class to verify that a component-info.xml class is valid.
 * 
 * @author <a href="pgier@redhat.com">Paul Gier</a>
 * 
 */
public class ComponentInfoReader
{

   /**
    * Read a component-info.xml file into a ComponentInfo object.
    * 
    * @param compInfoFile
    * @return
    */
   public static ComponentInfo parseComponentInfo(File compInfoFile) throws IOException, ParserConfigurationException, SAXException
   {
      if ( ! compInfoFile.exists() )
      {
         return null;
      }

      FileInputStream fis = new FileInputStream(compInfoFile);
      ComponentInfo compInfo = parseComponentInfo(fis);
      fis.close();
      
      return compInfo;
   }
   
   /**
    * Read a component-info.xml from an input stream into a ComponentInfo object.
    * 
    * @param compInfoFile
    * @return
    */
   public static ComponentInfo parseComponentInfo(InputStream compInfoIS) throws IOException, ParserConfigurationException, SAXException
   {
      ComponentInfo compInfo = new ComponentInfo();

      DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
      DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
      Document doc = docBuilder.parse(compInfoIS);

      // normalize text representation
      doc.getDocumentElement().normalize();
      Element projectElement = doc.getDocumentElement();

      NodeList componentNodes = projectElement.getElementsByTagName("component");
      Node componentNode = componentNodes.item(0);

      // General component information
      NamedNodeMap componentAttributes = componentNode.getAttributes();
      for (int i = 0; i < componentAttributes.getLength(); ++i)
      {
         Node attribute = componentAttributes.item(i);
         if (attribute.getNodeName().equals("id"))
         {
            compInfo.setComponentId(attribute.getNodeValue());
         }
         if (attribute.getNodeName().equals("licenseType"))
         {
            compInfo.setLicense(attribute.getNodeValue());
         }
         if (attribute.getNodeName().equals("version"))
         {
            compInfo.setVersion(attribute.getNodeValue());
         }
         if (attribute.getNodeName().equals("description"))
         {
            compInfo.setDescription( attribute.getNodeValue() );
         }
         if (attribute.getNodeName().equals("scm"))
         {
            compInfo.setScm( attribute.getNodeValue() );
         }
      }

      // Add artifactIds
      NodeList artifactNodes = projectElement.getElementsByTagName("artifact");
      for (int i = 0; i < artifactNodes.getLength(); ++i)
      {
         NamedNodeMap artifactAttr = artifactNodes.item(i).getAttributes();
         Node artifactIdNode = artifactAttr.item(0);
         compInfo.addArtifactId(artifactIdNode.getNodeValue());

      }

      // Add exports
      NodeList exportNodes = projectElement.getElementsByTagName("include");
      for (int i = 0; i < exportNodes.getLength(); ++i)
      {
         NamedNodeMap exportAttr = exportNodes.item(i).getAttributes();
         Node exportNode = exportAttr.item(0);
         compInfo.addExport(exportNode.getNodeValue());

      }

      // Add imports
      NodeList importNodes = projectElement.getElementsByTagName("import");
      for (int i = 0; i < importNodes.getLength(); ++i)
      {
         NamedNodeMap importAttr = importNodes.item(i).getAttributes();
         Node componentRefNode = importAttr.item(0);
         NodeList componentRefChildren = importNodes.item(i).getChildNodes();
         String compatVersions = "";
         for (int j = 0; j < componentRefChildren.getLength(); ++j)
         {
            Node child = componentRefChildren.item(j);
            if (child.getNodeName().equals("compatible"))
            {
               NamedNodeMap compatibleAttr = child.getAttributes();
               Node compatVersionsNode = compatibleAttr.item(0);
               compatVersions = compatVersionsNode.getNodeValue();
            }
         }

         compInfo.addImport(componentRefNode.getNodeValue(), compatVersions);
      }
         
      return compInfo;

   }
   
   /**
    * Verify a component-info.xml file using the xsd.
    * 
    * @param compInfoFilePath Path to the component-info.xml to be verified.
    */
   public static void verifyCompInfo(String compInfoFilePath) throws SAXException, java.io.FileNotFoundException
   {
      verifyCompInfo(new FileInputStream(compInfoFilePath));
   }

   /**
    * Verify a component-info.xml file using the xsd.
    * 
    * @param compInfoFilePath Path to the component-info.xml to be verified.
    */
   public static void verifyCompInfo(InputStream compInfo) throws SAXException
   {
      try
      {
         // Parse an XML document into a DOM tree.
         DocumentBuilder parser = DocumentBuilderFactory.newInstance().newDocumentBuilder();
         Document document = parser.parse(compInfo);

         // Create a SchemaFactory capable of understanding WXS schemas.
         SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);

         // Load a WXS schema, represented by a Schema instance.
         Source schemaFile = new StreamSource(ComponentInfoReader.class.getResourceAsStream("/org/jboss/maven/plugins/thirdparty/component-info.xsd"));
         Schema schema = factory.newSchema(schemaFile);

         // Create a Validator object, which can be used to validate
         // an instance document.
         Validator validator = schema.newValidator();

         // Validate the DOM tree.
         validator.validate(new DOMSource(document));
      }
      catch (ParserConfigurationException e)
      {
         e.printStackTrace();
      }
      catch (IOException e)
      {
         e.printStackTrace();
      }

   }

}
