/*
 * JBoss, the OpenSource J2EE webOS
 * 
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.maven.plugins.thirdparty;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.codehaus.plexus.util.IOUtil;

public class ComponentInfoWriter
{
   public final static String DEFAULT_TEMPLATE_FILENAME = "component-info-template.txt";
   
   public final static String DEFAULT_COMP_INFO_FILENAME = "component-info.xml";
      
   private static String componentInfoTemplate = "";
   
   public static String getComponentInfoTemplate()
   {
      return componentInfoTemplate;
   }

   public static void setComponentInfoTemplate(String theComponentInfoTemplate)
   {
      componentInfoTemplate = theComponentInfoTemplate;
   }
   
   /**
    * Load the component-info template file.
    * 
    */
   public static void loadTemplate() throws IOException
   {
      
      // Load template file
      InputStream templateInputStream = ComponentInfo.class.getResourceAsStream(DEFAULT_TEMPLATE_FILENAME);

      if (templateInputStream == null)
      {
         throw new IOException("Unable to load template file");
      }
      componentInfoTemplate = IOUtil.toString(templateInputStream);
   }
   
   /**
    * Write a componentInfo object to a file.
    * 
    * @param compInfo
    * @param outputFile
    * @return
    * @throws IOException
    */
   public static void writeComponentInfo( ComponentInfo compInfo, File outputFile ) throws IOException
   {
      FileWriter fw = new FileWriter( outputFile );
      fw.write( generateComponentInfo( compInfo) );
      fw.close();
   }
   
   public static String generateComponentInfo(ComponentInfo compInfo)
   {
      StringBuffer compInfoBuffer = new StringBuffer( componentInfoTemplate );
      
      // Evaluate the place holders in the template
      fillPlaceholder(compInfoBuffer, "project.name", compInfo.getProjectName());
      fillPlaceholder(compInfoBuffer, "component.id", compInfo.getComponentId());
      fillPlaceholder(compInfoBuffer, "project.version", compInfo.getVersion());
      fillPlaceholder(compInfoBuffer, "project.license", compInfo.getLicense());
      fillPlaceholder(compInfoBuffer, "project.description", compInfo.getDescription());
      fillPlaceholder(compInfoBuffer, "project.scm", compInfo.getScm());
      
      StringBuffer artifactLines = new StringBuffer();
      Iterator artifactIter = compInfo.getArtifactIds().iterator();
      while (artifactIter.hasNext())
      {
         artifactLines.append("    <artifact id=\"" + artifactIter.next() + "\"/>\n");
      }
      fillPlaceholder(compInfoBuffer, "artifacts", artifactLines.toString());
      
      // Create and set list of includes for export
      StringBuffer exportsString = new StringBuffer();
      Iterator exportIter = compInfo.getExports().iterator();
      while (exportIter.hasNext())
      {
         exportsString.append("      <include input=\"" + exportIter.next() + "\"/>\n");
      }
      fillPlaceholder(compInfoBuffer, "includes", exportsString.toString());

      // Generate the list of imports
      fillPlaceholder(compInfoBuffer, "imports", generateImportsString(compInfo));
      
      return compInfoBuffer.toString();
   }
   
   /** 
    * Creates the list of imports to include in the component-info.xml
    */
   private static String generateImportsString(ComponentInfo compInfo)
   {
      StringBuffer importsString = new StringBuffer();
      if (compInfo.getImports() != null)
      {
         Set componentNames = compInfo.getImports().keySet();
         Iterator iter = componentNames.iterator();
         while ( iter.hasNext() )
         {
            ComponentInfo component = (ComponentInfo)iter.next();
            importsString.append("    <import componentref=\"" + component.toString().replace('.', '/') + "\">\n");
            String componentVersions = compInfo.getImports().get(component).toString();
            String [] versions = componentVersions.split(",");
            for(int i=0; i<versions.length; ++i)
            {
               importsString.append("      <compatible version=\"" + versions[i].trim() + "\"/>\n");
            }
            importsString.append("    </import>\n");
         }
      }
      return importsString.toString();
   }

   /**
    * Generate an entry for the libraries.ent file based on the given componentInfo object
    * 
    * @param compInfo
    * @return
    */
   public static String getLibrariesEntEntry(ComponentInfo compInfo)
   {
      StringBuffer libEnt = new StringBuffer();
      String libEntCompId = compInfo.getLibrariesEntCompId();
      libEnt.append( "<!-- " + compInfo.getComponentId() + " -->\n");
      libEnt.append( "<property name=\"" + libEntCompId + ".root\" value=\"${project.thirdparty}/" + compInfo.getComponentId().replace('.', '/') + "\"/>\n" );
      libEnt.append( "<property name=\"" + libEntCompId + ".lib\" value=\"${" + libEntCompId + ".root}/lib/\"/>\n" );
      libEnt.append( "<property name=\"" + libEntCompId + ".resources\" value=\"${" + libEntCompId + ".root}/resources/\"/>\n" );
      libEnt.append( "<path id=\"" + libEntCompId + ".classpath\">\n");
      List sortedArtifacts = new ArrayList( compInfo.getArtifactIds() );
      Collections.sort( sortedArtifacts );
      for (int i=0; i<sortedArtifacts.size(); ++i )
      {
         String currArtifact = (String)sortedArtifacts.get( i );
         libEnt.append( "   <pathelement path=\"${" + libEntCompId + ".lib}/" + currArtifact + "\"/>\n");
      }
      libEnt.append( "</path>\n\n" );
     
      return libEnt.toString();
   }

   /**
    * Replace all instances of placeholder with value in the given buffer.
    * For example if placeholder is "project.name", this method will search
    * the buffer for "${project.name}" and replace it with the value
    * provided.
    * 
    * @param buffer The buffer to be modified
    * @param var The name of the variable to be evaluated.
    * @param value The replacement string.
    */
   public static void fillPlaceholder(StringBuffer buffer, String placeholder, String value)
   {
      int start = 0;
      while ((start = buffer.indexOf("${" + placeholder + "}")) != -1)
      {
         int end = start + ("${" + placeholder + "}").length();
         buffer.replace(start, end, value);
      }
   }
   
   

}
