package org.jboss.maven.plugins.thirdparty;

import java.io.File;
import java.io.IOException;
import java.io.Writer;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.deployer.ArtifactDeployer;
import org.apache.maven.artifact.deployer.ArtifactDeploymentException;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadata;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.artifact.ProjectArtifactMetadata;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.WriterFactory;
import org.jboss.maven.plugins.thirdparty.util.JarUtil;

/**
 * Mojo for taking a thirdparty deployment (component-info.xml, lib/*.jar, etc..) and deploying it to
 * a maven repository.  This mojo functions in a similar way to the deploy-file goal of the 
 * <a href="http://maven.apache.org/plugins/maven-deploy-plugin/">maven-deploy-plugin</a>.
 * 
 * @goal maven-deploy
 * @requiresProject false
 * 
 */
public class MavenDeployMojo extends AbstractMojo
{

   public static final Map KNOWN_PACKAGING_TYPES = new HashMap();
   
   // Initialize the known packaging types.
   static 
   {
      KNOWN_PACKAGING_TYPES.put( "jar", "jar");
      KNOWN_PACKAGING_TYPES.put( "war", "war");
      KNOWN_PACKAGING_TYPES.put( "zip", "zip");
   }
   
   /**
    * @parameter expression="${component.org.apache.maven.artifact.deployer.ArtifactDeployer}"
    * @readonly
    */
   private ArtifactDeployer deployer;

   /**
    * @parameter expression="${localRepository}"
    * @readonly
    */
   private ArtifactRepository localRepository;

   /**
    * GroupId of the artifact to be deployed.  Retrieved from POM file if specified.
    *
    * @parameter expression="${groupId}"
    */
   private String groupId;

   /**
    * Directory to search for component-info.xml.
    * Defaults to current directory.
    *
    * @parameter expression="${componentDir}" default-value="."
    * @required
    */
   private File componentDir;

   /**
    * Version of the component to be deployed.  Defaults to version in component-info.xml
    *
    * @parameter expression="${version}"
    */
   private String version;

   /**
    * Description passed to a generated POM file (in case of generatePom=true)
    *
    * @parameter expression="${generatePom.description}"
    */
   private String description;

   /**
    * Server Id to map on the &lt;id&gt; under &lt;server&gt; section of settings.xml
    * In most cases, this parameter will be required for authentication.
    *
    * @parameter expression="${repositoryId}" default-value="remote-repository"
    */
   private String repositoryId;

   /**
    * The type of remote repository layout to deploy to. Try <i>legacy</i> for 
    * a Maven 1.x-style repository layout.
    * 
    * @parameter expression="${repositoryLayout}" default-value="default"
    */
   private String repositoryLayout;

   /**
    * Map that contains the layouts
    *
    * @component role="org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout"
    */
   private Map repositoryLayouts;

   /**
    * URL where the artifact will be deployed. <br/>
    * ie ( file://C:\m2-repo or scp://host.com/path/to/repo )
    *
    * @parameter expression="${url}"
    * @required
    */
   private String url;

   /**
    * License to use for the components.  Will default to the license in the 
    * component-info.xml
    *
    * @parameter expression="${license}"
    */
   private String license;

   /**
    * Component used to create an artifact
    *
    * @component
    */
   private ArtifactFactory artifactFactory;

   /**
    * Component used to create a repository
    *
    * @component
    */
   private ArtifactRepositoryFactory repositoryFactory;

   /**
    * Whether to deploy snapshots with a unique version or not.
    *
    * @parameter expression="${uniqueVersion}" default-value="true"
    */
   private boolean uniqueVersion;

   /**
    * List of file name pattern to include in a resources jar. These
    * should be "/" separated path relative to the component directory.
    * When downloaded by the build-thirdparty plugin, these resources
    * will be extracted into the componentDirectory.
    * 
    * By default the plugin will look for directories called 
    * "resources" and "bin" and include them in the resources
    * jar if they exist.
    *
    * @parameter
    */
   private String [] resourceIncludes;
   
   /**
    * Patterns that should not be included in the resources
    *
    * @parameter
    */
   private String [] resourceExcludes;
   

   public void execute() throws MojoExecutionException
   {

      if (!componentDir.isDirectory())
      {
         throw new MojoExecutionException(componentDir.getPath() + " does not exist or is not a directory.");
      }
      
      File compInfoFile = new File(componentDir, "component-info.xml");
      if ( ! compInfoFile.exists() )
      {
         getLog().error( "Unable to locate component-info.xml : " + compInfoFile );
         throw new MojoExecutionException( "Unable to locate component-info.xml" );
      }
      
      ArtifactRepositoryLayout layout;

      layout = (ArtifactRepositoryLayout) repositoryLayouts.get(repositoryLayout);

      ArtifactRepository deploymentRepository = repositoryFactory.createDeploymentArtifactRepository(repositoryId, url,
            layout, uniqueVersion);

      String protocol = deploymentRepository.getProtocol();

      if ("".equals(protocol) || protocol == null)
      {
         throw new MojoExecutionException("No transfer protocol found. This could be caused " +
         		"by an invalid url: " + url);
      }

      
      // Read the component-info file
      ComponentInfo compInfo = null;
      try 
      {
         compInfo = ComponentInfoReader.parseComponentInfo( compInfoFile );
      } 
      catch ( Exception e )
      {
         throw new MojoExecutionException( "Unable to read component-info.xml", e );
      }
      
      if ( groupId == null )
      {
         groupId = compInfo.getComponentId().replace('/', '.');
      }
      if ( version == null )
      {
         version = compInfo.getVersion();
      }
      if ( license == null )
      {
         license = compInfo.getLicense();
      }
      
      File artifactLibDir = new File(componentDir, "lib");
      
      List artifacts = compInfo.getArtifactIds();
      Iterator artifactIter = artifacts.iterator();
      
      while ( artifactIter.hasNext())
      {
         String artifactId = (String)artifactIter.next();
         
         File artifactFile = new File(artifactLibDir, artifactId);
         
         if ( ! artifactFile.exists() )
         {
            getLog().warn("Artifact file does not exist: " + artifactFile);
            getLog().warn("This file will be skipped.");
            continue;
         }
         
         // Create the artifact metadata
         int extIndex = artifactId.lastIndexOf('.');
         String mavenArtifactId = artifactId.substring(0, extIndex);
         
         // Handle special case where version number is in the artifactId.
         String artifactVersion = this.version;
         if ( mavenArtifactId.contains("-") )
         {
            int dashIndex = mavenArtifactId.lastIndexOf('-');
            if ( Character.isDigit(mavenArtifactId.charAt(dashIndex + 1)))
            {
               artifactVersion = mavenArtifactId.substring(dashIndex + 1);
               mavenArtifactId = mavenArtifactId.substring(0, dashIndex);
            }
         }
         
         String fileExtension = artifactId.substring(extIndex + 1);
         
         if ( ! KNOWN_PACKAGING_TYPES.containsKey( fileExtension ) )
         {
            getLog().warn( "Unknown packaging: " + fileExtension );
            getLog().warn( "Skipping deployment of: " + artifactId );
            continue;
         }
         String packaging = (String)KNOWN_PACKAGING_TYPES.get(fileExtension);
         
         String classifier = null;
         
         if (mavenArtifactId.endsWith( "-src") || mavenArtifactId.endsWith( "-sources" ) )
         {
            // If the version is a snapshot, we have to skip deploying source jars
            // because the source jar will appear as the only deployed jar if it gets deployed after the main jar.
            if ( artifactVersion.endsWith( "-SNAPSHOT" ) )
            {
               getLog().warn( "Skipping deployment of source jar because of SNAPSHOT version: " + artifactId );
               continue;
            }
            classifier = "sources";
            mavenArtifactId = mavenArtifactId.replace("-sources", "").replace("-src", "");
            packaging = "jar";
         }
         
         Artifact artifact = artifactFactory.createArtifactWithClassifier(groupId, mavenArtifactId, artifactVersion, packaging,
               classifier);
         ArtifactMetadata metadata = new ProjectArtifactMetadata(artifact, generatePomFile(mavenArtifactId, artifactVersion, packaging));
         artifact.addMetadata(metadata);
         
         try
         {
            getDeployer().deploy(artifactFile, artifact, deploymentRepository, getLocalRepository());
         }
         catch (ArtifactDeploymentException e)
         {
            throw new MojoExecutionException(e.getMessage(), e);
         }
      }
      
      // Try to deploy a jar with the resources folder if it exists.
      getLog().info( "Time to deploy resources: "  );
      deployResources( deploymentRepository );

   }
   
   /**
    * This method attempts to deploy resources directories if they exist.
    * 
    * @param deploymentRepository
    * @throws MojoExecutionException
    */
   private void deployResources(ArtifactRepository deploymentRepository) throws MojoExecutionException
   {

      //List<JarFileEntry> jarContents = new ArrayList<JarFileEntry>();
      if ( resourceIncludes == null )
      {
         resourceIncludes = new String[2];
         resourceIncludes[0] = "resources/**";
         resourceIncludes[1] = "bin/**";
      }

      List jarEntries = null;
      try 
      {
         jarEntries = JarUtil.getJarEntries( componentDir, resourceIncludes, resourceExcludes );
      }
      catch ( IOException e )
      {
         getLog().warn( "Problem reading resources: " + e);
      }
      
      if ( jarEntries==null || jarEntries.isEmpty() )
      {
         return;
      }
      
      getLog().info( "Creating resources jar..." );
      
      String mavenArtifactId = "resources";
      String packaging = "jar";
      String classifier = null;
                  
      Artifact artifact = artifactFactory.createArtifactWithClassifier(this.groupId, mavenArtifactId, 
            this.version, packaging, classifier);
      ArtifactMetadata metadata = new ProjectArtifactMetadata(artifact, generatePomFile(mavenArtifactId, 
            this.version, packaging));
      artifact.addMetadata(metadata);
      
      
      try 
      {
         String tempJarFileName = "resources";
         File tempJarFile = File.createTempFile(tempJarFileName, "jar");
         JarUtil.writeJarFile(jarEntries, tempJarFile);
         
         getDeployer().deploy(tempJarFile, artifact, deploymentRepository, getLocalRepository());
      }
      catch ( IOException e )
      {
         getLog().warn( "Problem while writing jar file: " + e);
      }
      catch (ArtifactDeploymentException e)
      {
         throw new MojoExecutionException(e.getMessage(), e);
      }
      
   }
      
   private File generatePomFile( String artifactId, String artifactVersion, String packaging ) throws MojoExecutionException
   {
      Writer fw = null;
      try
      {
         File tempFile = File.createTempFile("mvninstall", ".pom");
         tempFile.deleteOnExit();

         Model model = new Model();
         model.setModelVersion("4.0.0");
         model.setGroupId(groupId);
         model.setArtifactId(artifactId);
         model.setVersion(artifactVersion);
         model.setPackaging(packaging);
         model.setDescription(description);

         fw = WriterFactory.newXmlWriter(tempFile);
         new MavenXpp3Writer().write(fw, model);

         return tempFile;
      }
      catch (IOException e)
      {
         throw new MojoExecutionException("Error writing temporary pom file: " + e.getMessage(), e);
      }
      finally
      {
         IOUtil.close(fw);
      }
   }

   void setGroupId(String groupId)
   {
      this.groupId = groupId;
   }

   void setVersion(String version)
   {
      this.version = version;
   }

   String getGroupId()
   {
      return groupId;
   }

   String getVersion()
   {
      return version;
   }

   public ArtifactDeployer getDeployer()
   {
      return deployer;
   }

   public void setDeployer(ArtifactDeployer deployer)
   {
      this.deployer = deployer;
   }

   public ArtifactRepository getLocalRepository()
   {
      return localRepository;
   }

   public void setLocalRepository(ArtifactRepository localRepository)
   {
      this.localRepository = localRepository;
   }

}
