package org.jboss.maven.plugins.thirdparty;

import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.deployer.ArtifactDeployer;
import org.apache.maven.artifact.deployer.ArtifactDeploymentException;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadata;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout;
import org.apache.maven.model.License;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.artifact.ProjectArtifactMetadata;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.ReaderFactory;
import org.codehaus.plexus.util.WriterFactory;
import org.jboss.maven.plugins.thirdparty.util.FileUtil;
import org.jboss.maven.plugins.thirdparty.util.JarUtil;

/**
 * Mojo for taking a thirdparty deployment (component-info.xml, lib/*.jar, etc..) and deploying it to
 * a maven repository.  This mojo functions in a similar way to the deploy-file goal of the 
 * <a href="http://maven.apache.org/plugins/maven-deploy-plugin/">maven-deploy-plugin</a>.
 * 
 * @goal maven-deploy
 * @requiresProject false
 * 
 */
public class MavenDeployMojo extends AbstractMojo
{

   public static final Set KNOWN_EXTENSIONS = new HashSet();
   
   // Initialize the known packaging types.
   static 
   {
      KNOWN_EXTENSIONS.add( "jar" );
      KNOWN_EXTENSIONS.add( "war" );
      KNOWN_EXTENSIONS.add( "zip" );
      KNOWN_EXTENSIONS.add( "sar" );
   }
   
   /**
    * The standard Maven artifact deployer component
    * @component
    */
   private ArtifactDeployer deployer;

   /**
    * @parameter expression="${localRepository}"
    * @readonly
    */
   private ArtifactRepository localRepository;

   /**
    * GroupId of the artifact to be deployed.  Retrieved from POM file if specified.
    *
    * @parameter expression="${groupId}"
    */
   private String groupId;

   /**
    * Directory to search for component-info.xml.
    * Defaults to current directory.
    *
    * @parameter expression="${componentDir}" default-value="."
    * @required
    */
   private File componentDir;

   /**
    * Version of the component to be deployed.  Defaults to version in component-info.xml
    *
    * @parameter expression="${version}"
    */
   private String version;

   /**
    * Description passed to a generated POM file (in case of generatePom=true)
    *
    * @parameter expression="${generatePom.description}"
    */
   private String description;

   /**
    * Server Id to map on the &lt;id&gt; under &lt;server&gt; section of settings.xml
    * In most cases, this parameter will be required for authentication.
    *
    * @parameter expression="${repositoryId}" default-value="remote-repository"
    */
   private String repositoryId;

   /**
    * The type of remote repository layout to deploy to. Try <i>legacy</i> for 
    * a Maven 1.x-style repository layout.
    * 
    * @parameter expression="${repositoryLayout}" default-value="default"
    */
   private String repositoryLayout;

   /**
    * Map that contains the layouts
    *
    * @component role="org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout"
    */
   private Map repositoryLayouts;

   /**
    * URL where the artifact will be deployed. <br/>
    * ie ( file://C:\m2-repo or scp://host.com/path/to/repo )
    *
    * @parameter expression="${url}"
    * @required
    */
   private String url;

   /**
    * License to use for the components.  Will default to the license in the 
    * component-info.xml
    *
    * @parameter expression="${license}"
    */
   private String license;

   /**
    * Component used to create an artifact
    *
    * @component
    */
   private ArtifactFactory artifactFactory;

   /**
    * Component used to create a repository
    *
    * @component
    */
   private ArtifactRepositoryFactory repositoryFactory;

   /**
    * Whether to deploy snapshots with a unique version or not.
    *
    * @parameter expression="${uniqueVersion}" default-value="true"
    */
   private boolean uniqueVersion;

   /**
    * List of file name pattern to include in a resources jar. These
    * should be "/" separated path relative to the component directory.
    * When downloaded by the build-thirdparty plugin, these resources
    * will be extracted into the componentDirectory.
    * 
    * By default the plugin will look for directories called 
    * "resources" and "bin" and include them in the resources
    * jar if they exist.
    *
    * @parameter
    */
   private String [] resourceIncludes;
   
   /**
    * Patterns that should not be included in the resources
    *
    * @parameter
    */
   private String [] resourceExcludes;
   
   /**
    * A comma separated list of strings to recognize as artifact classifiers.
    *
    * @parameter default-value="sources,tests" expression="${classifiers}"
    */
   private String classifiers;
   
   /**
    * The list of recognized classifiers.
    */
   private String [] classifierList;
   
   /**
    * Main plugin execution
    */
   public void execute() throws MojoExecutionException
   {

      // Check that the componentDir exists and contains a component-info.xml file.
      if (!componentDir.isDirectory())
      {
         throw new MojoExecutionException(componentDir.getPath() + " does not exist or is not a directory.");
      }
      
      File compInfoFile = new File(componentDir, "component-info.xml");
      if ( ! compInfoFile.exists() )
      {
         getLog().error( "Unable to locate component-info.xml : " + compInfoFile );
         throw new MojoExecutionException( "Unable to locate component-info.xml" );
      }
      
      // Check that we have a valid repository URL
      ArtifactRepositoryLayout layout = (ArtifactRepositoryLayout) repositoryLayouts.get( repositoryLayout );

      ArtifactRepository deploymentRepository = repositoryFactory.createDeploymentArtifactRepository( repositoryId, url,
            layout, uniqueVersion );

      String protocol = deploymentRepository.getProtocol();

      if ("".equals(protocol) || protocol == null)
      {
         throw new MojoExecutionException("No transfer protocol found. This could be caused " +
         		"by an invalid url: " + url);
      }

      // Read the component-info file
      ComponentInfo compInfo = null;
      try 
      {
         compInfo = ComponentInfoReader.parseComponentInfo( compInfoFile );
      } 
      catch ( Exception e )
      {
         throw new MojoExecutionException( "Unable to read component-info.xml", e );
      }
      
      File componentLibDir = new File(componentDir, "lib");
      
      // Parse the comma separated list of classifiers.
      classifiers = classifiers.trim();
      classifierList = classifiers.split("\\s*\\,\\s*");
      
      Map pomToArtifactMap = this.mapPomsToArtifacts( compInfo, componentLibDir );
      
      // Iterate over the poms and deploy associated artifacts
      Iterator pomIter = pomToArtifactMap.keySet().iterator();
      while ( pomIter.hasNext())
      {
         File pomFile = (File)pomIter.next();
         List pomArtifacts = (List)pomToArtifactMap.get(pomFile);
                  
         Model model = readModel(pomFile);
         
         // Might need to get groupId or version from parent config
         if ( model.getGroupId() == null )
         {
            model.setGroupId( model.getParent().getGroupId() );
         }
         if ( model.getVersion() == null )
         {
            model.setVersion( model.getParent().getVersion() );
         }
         
         // Override artifact information with command line values if applicable
         boolean modelChanged = false;
         if ( this.groupId != null )
         {
            model.setGroupId( this.groupId );
            modelChanged = true;
         }
         if ( this.version != null )
         {
            model.setVersion( this.version );
            modelChanged = true;
         }
         if ( this.description != null )
         {
            model.setDescription( this.description );
            modelChanged = true;
         }
         
         if ( modelChanged || ( ! pomFile.exists() ) )
         {
           pomFile = writeModel( model );
         }
         
         if ( pomArtifacts.isEmpty() )
         {
            String packaging = "pom";
            String classifier = null;
            Artifact artifact = artifactFactory.createArtifactWithClassifier(model.getGroupId(), model.getArtifactId(),
                  model.getVersion(), packaging, classifier);
            ArtifactMetadata metadata = new ProjectArtifactMetadata(artifact, pomFile );
            artifact.addMetadata(metadata);
            
            try
            {
               getDeployer().deploy(pomFile, artifact, deploymentRepository, getLocalRepository());
            }
            catch (ArtifactDeploymentException e)
            {
               throw new MojoExecutionException(e.getMessage(), e);
            }
         }
         else
         {
            // Deploy each artifact
            for ( int i=0; i<pomArtifacts.size(); ++i )
            {
               File artifactFile = (File)pomArtifacts.get( i );
               String packaging = FileUtil.getFileExtension( artifactFile.getName() );
               String baseArtifactName = FileUtil.removeFileExtension( artifactFile.getName() );
               
               String classifier = null;
               
               // Check if we have an artifact with a classifier
               for ( int j=0; j<classifierList.length; ++j )
               {
                  if (baseArtifactName.endsWith( "-" + classifierList[j] ) )
                  {
                     classifier = classifierList[j];
                  }
               }
               
               if ( classifier != null )
               {
                  // If the version is a snapshot, we have to skip deploying source jars
                  // because the source jar will appear as the only deployed jar if it gets deployed after the main jar.
                  if ( model.getVersion().endsWith( "-SNAPSHOT" ) )
                  {
                     getLog().info( "Skipping deployment of source jar because of SNAPSHOT version: " + artifactFile.getName() );
                     continue;                     
                  }
               }
               
               /*// Check if we are dealing with a source jar
               if (baseArtifactName.endsWith( "-src") || baseArtifactName.endsWith( "-sources" ) )
               {
                  // If the version is a snapshot, we have to skip deploying source jars
                  // because the source jar will appear as the only deployed jar if it gets deployed after the main jar.
                  if ( model.getVersion().endsWith( "-SNAPSHOT" ) )
                  {
                     getLog().info( "Skipping deployment of source jar because of SNAPSHOT version: " + artifactFile.getName() );
                     continue;                     
                  }
                  classifier = "sources";
                  baseArtifactName = baseArtifactName.replace("-sources", "").replace("-src", "");
                  packaging = "jar";
               }*/
               
               Artifact artifact = artifactFactory.createArtifactWithClassifier(model.getGroupId(), model.getArtifactId(),
                     model.getVersion(), packaging, classifier);
               ArtifactMetadata metadata = new ProjectArtifactMetadata(artifact, pomFile );
               artifact.addMetadata(metadata);
               
               try
               {
                  getDeployer().deploy(artifactFile, artifact, deploymentRepository, getLocalRepository());
               }
               catch (ArtifactDeploymentException e)
               {
                  throw new MojoExecutionException(e.getMessage(), e);
               }
            }

         }
      }
      
      // Try to deploy the resources folder as a jar.
      deployResources( compInfo, deploymentRepository );

   }
   
   /**
    * Map the POMs to associated artifacts (if any) in a component lib dir
    * 
    * @param artifactLibDir
    * @return
    */
   private Map mapPomsToArtifacts( ComponentInfo compInfo, File componentLibDir )
       throws MojoExecutionException
   {
      File [] fileList = componentLibDir.listFiles();
      
      // Separate poms from artifacts
      Map pomToArtifactMap = new HashMap();
      Stack artifactStack = new Stack();
      for ( int i=0; i<fileList.length; ++i )
      {
         if ( fileList[i].isDirectory() )
         {
            continue;
         }
         else if ( fileList[i].getName().endsWith( ".pom" ) )
         {
            pomToArtifactMap.put( fileList[i], new ArrayList() );
         }
         else
         {
            String fileExtension = FileUtils.getExtension( fileList[i].getName() );
            if ( KNOWN_EXTENSIONS.contains( fileExtension ) )
            {
               artifactStack.add( fileList[i] );
            }
            else
            {
               getLog().info( "Skipping file: " + fileList[i].getName() );
               getLog().info( "Unknown packaging: " + fileExtension );
            }
         }
      }
      
      // Map artifacts to poms
      while ( ! artifactStack.isEmpty() )
      {
         File nextArtifact = (File)artifactStack.pop();
         String artifactName = FileUtil.removeFileExtension( nextArtifact.getName() );
         // Remove qualifier if exists
         for ( int i=0; i<classifierList.length; ++i )
         {
            artifactName = artifactName.replace( "-"+classifierList[i], "" );
         }

         // Handle special case where version number is in the artifactId.
         if ( artifactName.contains("-") )
         {
            int dashIndex = artifactName.lastIndexOf('-');
            if ( Character.isDigit(artifactName.charAt(dashIndex + 1)))
            {
               artifactName = artifactName.substring(0, dashIndex);
            }
         }
         
         String pomName = artifactName + ".pom";
         File pomFile = new File( componentLibDir, pomName);
         
         if ( ! pomToArtifactMap.keySet().contains( pomFile ) )
         {
            pomFile = this.generatePomFile( compInfo, nextArtifact );
            ArrayList pomArtifactList = new ArrayList();
            pomArtifactList.add( nextArtifact );
            pomToArtifactMap.put( pomFile, pomArtifactList );
         }
         List artifactList = (List)pomToArtifactMap.get( pomFile );
         artifactList.add( nextArtifact );

      }
      return pomToArtifactMap;
   }
   
   /**
    * Extract the Model from the specified file.
    *
    * @param pomFile
    * @return
    * @throws MojoExecutionException if the file doesn't exist of cannot be read.
    */
   protected Model readModel( File pomFile )
       throws MojoExecutionException
   {

       if ( !pomFile.exists() )
       {
           throw new MojoExecutionException( "Specified pomFile does not exist" );
       }

       Reader reader = null;
       try
       {
           reader = ReaderFactory.newXmlReader( pomFile );
           MavenXpp3Reader modelReader = new MavenXpp3Reader();
           return modelReader.read( reader );
       }
       catch ( Exception e )
       {
           throw new MojoExecutionException( "Error reading specified POM file: " + e.getMessage(), e );
       }
       finally
       {
           IOUtil.close( reader );
       }
   }
   
   /**
    * Write a POM model to a file
    * 
    * @param model
    * @return
    * @throws MojoExecutionException
    */
   public File writeModel ( Model model ) throws MojoExecutionException
   {
      Writer fw = null;
      try 
      {
         File tempFile = File.createTempFile(model.getArtifactId(), ".pom");
         tempFile.deleteOnExit();
   
         fw = WriterFactory.newXmlWriter(tempFile);
         new MavenXpp3Writer().write(fw, model);
   
         return tempFile;
      }
      catch (IOException e)
      {
         throw new MojoExecutionException("Error writing temporary pom file: " + e.getMessage(), e);
      }
      finally
      {
         IOUtil.close(fw);
      }
   }

   /**
    * This method attempts to deploy resources directories if they exist.
    * 
    * @param deploymentRepository
    * @throws MojoExecutionException
    */
   private void deployResources(ComponentInfo compInfo, ArtifactRepository deploymentRepository) throws MojoExecutionException
   {

      //List<JarFileEntry> jarContents = new ArrayList<JarFileEntry>();
      if ( resourceIncludes == null )
      {
         resourceIncludes = new String[2];
         resourceIncludes[0] = "resources/**";
         resourceIncludes[1] = "bin/**";
      }

      List jarEntries = null;
      try 
      {
         jarEntries = JarUtil.getJarEntries( componentDir, resourceIncludes, resourceExcludes );
      }
      catch ( IOException e )
      {
         getLog().warn( "Problem reading resources: " + e);
      }
      
      if ( jarEntries==null || jarEntries.isEmpty() )
      {
         return;
      }
      
      getLog().info( "Found resources, creating jar for deployment" );
      
      String mavenGroupId = groupId;
      if ( mavenGroupId == null )
      {
         mavenGroupId = compInfo.getComponentId().replace('/', '.');
      }
      String mavenArtifactId = "resources";
      String packaging = "jar";
      String classifier = null;
      String mavenVersion = this.version;
      if ( mavenVersion == null )
      {
         mavenVersion = compInfo.getVersion();
      }
                  
      Model model = generateModel( mavenGroupId, mavenArtifactId, mavenVersion, packaging, license );
      Artifact artifact = artifactFactory.createArtifactWithClassifier( model.getGroupId(), model.getArtifactId(), 
            model.getVersion(), model.getPackaging(), classifier );
      ArtifactMetadata metadata = new ProjectArtifactMetadata( artifact, writeModel( model ) );
      artifact.addMetadata( metadata );
      
      
      try 
      {
         String tempJarFileName = "resources";
         File tempJarFile = File.createTempFile(tempJarFileName, "jar");
         JarUtil.writeJarFile(jarEntries, tempJarFile);
         
         getDeployer().deploy(tempJarFile, artifact, deploymentRepository, getLocalRepository());
      }
      catch ( IOException e )
      {
         getLog().warn( "Problem while writing jar file: " + e);
      }
      catch (ArtifactDeploymentException e)
      {
         throw new MojoExecutionException(e.getMessage(), e);
      }
      
   }
   
   /**
    * Get the name of the file without the extention or the -source qualifier.
    * 
    * @param artifactFileName
    * @return
    */
   public String getArtifactBaseName( String artifactFileName )
   {
      artifactFileName = FileUtil.removeFileExtension( artifactFileName );
      artifactFileName = FileUtil.removeSourceQualifier( artifactFileName );
      return artifactFileName;
   }
      
   /**
    * Create a pom file for the given artifactFile and component info
    * 
    * @param compInfo
    * @param artifactFile
    * @return
    * @throws MojoExecutionException
    */
   private File generatePomFile(ComponentInfo compInfo, File artifactFile ) throws MojoExecutionException
   {
      String groupId = compInfo.getComponentId().replace('/', '.');
      String artifactId = getArtifactBaseName( artifactFile.getName() );
      String artifactVersion = compInfo.getVersion();
      String packaging = FileUtils.getExtension( artifactFile.getName() );
      String license = compInfo.getLicense();
      
      Model model = generateModel(groupId, artifactId, artifactVersion, packaging, license);
      File pomFile = this.writeModel( model );
      return pomFile;
   }
   
   private Model generateModel(String groupId, String artifactId, String artifactVersion, String packaging,
         String license) throws MojoExecutionException
   {
      Model model = new Model();
      model.setModelVersion("4.0.0");
      model.setGroupId(groupId);
      model.setArtifactId(artifactId);
      model.setVersion(artifactVersion);
      model.setPackaging(packaging);
      model.setDescription(description);

      if (license != null)
      {
         License theLicense = new License();
         theLicense.setName(license);
         model.addLicense(theLicense);
      }

      return model;
   }
   
   void setGroupId(String groupId)
   {
      this.groupId = groupId;
   }

   void setVersion(String version)
   {
      this.version = version;
   }

   String getGroupId()
   {
      return groupId;
   }

   String getVersion()
   {
      return version;
   }

   public ArtifactDeployer getDeployer()
   {
      return deployer;
   }

   public void setDeployer(ArtifactDeployer deployer)
   {
      this.deployer = deployer;
   }

   public ArtifactRepository getLocalRepository()
   {
      return localRepository;
   }

   public void setLocalRepository(ArtifactRepository localRepository)
   {
      this.localRepository = localRepository;
   }

}
