package org.jboss.maven.plugins.thirdparty;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.License;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.FileUtils;
import org.xml.sax.SAXException;

/**
 * Maven plugin for deploying maven generated jars to the buildmagic repository.  By default
 * it will bind to the deploy phase.
 * 
 * @phase deploy
 * @goal thirdparty-deploy
 * 
 */
public class ThirdpartyDeployMojo extends AbstractMojo
{

   private static String fileSep = System.getProperty("file.separator");

   /**
    * The Maven Project Object
    *
    * @parameter expression="${project}"
    * @readonly
    */
   protected MavenProject project;

   /**
    * Source control information.
    * @parameter expression="${project.scm}"
    * @readonly
    */
   private org.apache.maven.model.Scm scm;

   /**
    * License information from the pom.xml.
    * @parameter expression="${project.licenses}"
    * @readonly
    */
   private List pomLicenses;

   /**
    * Name of license to add to component-info.xml.  If no value is specified,
    * then the first license from the pom.xml will be used.
    * @parameter
    */
   private String license;

   /**
    * Boolean that determines if the artifact version should be included in the
    * deployed jar files.
    * @parameter default-value="false"
    */
   private boolean removeArtifactVersion;

   /**
    * Maven project properties.
    *
    * @parameter expression="${project.properties}"
    * @readonly
    */
   protected Properties projectProperties;

   /**
    * This is the location that the jboss repository (non-maven) files will be copied to.
    * @parameter expression="${thirdpartyRepositoryDir}"
    * @required
    */
   private String thirdpartyRepositoryDir;

   /**
    * If set to true, this will tell the plugin to generate a listing of imports based on the
    * project dependencies.  Defaults to false;
    * @parameter
    */
   private boolean generateImports;

   /**
    * This is the map of components that this project depends on.  If generateImports is set
    * to true, the generated imports will be appended to the imports specified here.
    * @parameter
    */
   private Map imports;

   /**
    * This allows the maven artifactId to be overridden.  For example, the pom artifactId
    * might be "jboss-metadata", but the jboss repository artifactId should just be "metadata".
    * 
    * @parameter default-value="${project.artifactId}"
    */
   private String artifactId;

   /**
    * The componentId of the artifact(s) to be deployed.  For example, the pom groupId
    * might be "org.jboss.metadata", but the desired thirdparty repository componentId is "jboss.metadata".
    * 
    * @parameter default-value="${project.groupId}"
    */
   private String componentId;

   /**
    * This allows the maven version to be overridden.
    * 
    * @parameter default-value="${project.version}"
    */
   private String componentVersion;

   /**
    * This is the list of artifacts exported from this project.
    * By default the project artifact will be included in this list.
    * If there is no project artifact, this plugin will look for an attached
    * artifact that matches the project name, and attempt to attach it.
    * The list of exports can manually be set with the following configuration.
    * 
    * <pre>
    * &lt;exports&gt;
    * &nbsp;&lt;param&gt;file1.jar&lt;/param&gt;
    * &nbsp;&lt;param&gt;file2.jar&lt;/param&gt;
    * &lt;/exports&gt;
    * </pre>
    * 
    * @parameter
    */
   private Set exports;
   
   /**
    * This parameter provides an option to skip execution of the plugin.  This parameter
    * can be set from the command line using the property maven.jboss-deploy.skip.
    * 
    * @parameter expression="${maven.jboss-deploy.skip}"
    */
   private boolean skip;

   /**
    * Main execution path of the plugin.  Generates component-info.xml, and copies jar files to repository location.
    */
   public void execute() throws MojoExecutionException
   {
      this.getLog().debug("Executing JBoss deploy plugin");
      
      // Check if we should skip execution of this plugin
      if ( skip )
      {
         this.getLog().info("Skipping execution of jboss-deploy plugin");
         return;
      }
      
      // Check that there is at least one artifact to deploy
      if ((project.getArtifact() == null || project.getArtifact().getFile() == null)
            && project.getAttachedArtifacts().size() == 0)
      {
         this.getLog().info("No artifacts to deploy for project:" + project.getName());
         return;
      }

      if (exports == null) 
      {
         exports = new HashSet();
      }
      // Create a componentInfo object to represent the artifact(s).
      ComponentInfo compInfo = new ComponentInfo();
      compInfo.setComponentId( componentId );
      compInfo.setVersion( componentVersion );

      // Initialize some local variables
      File componentBaseDir = new File(thirdpartyRepositoryDir, compInfo.getComponentId() );
      System.out.println( componentBaseDir );
      File componentVersionDir = new File( componentBaseDir, componentVersion );
      System.out.println( componentVersionDir );
      File componentLibDir = new File( componentVersionDir, "lib" );

      if (license != null && ( ! license.equals(""))) 
      {
         compInfo.setLicense(license);
      }
      else if (pomLicenses.size() == 0)
      {
         getLog().warn("No license specified.  Default lgpl license will be used.");
         compInfo.setLicense("lgpl");
      }
      else
      {
         License firstLicense = (License) pomLicenses.get(0);
         compInfo.setLicense(firstLicense.getName());
      }
      
      if (project.getDescription() == null || project.getDescription().equals(""))
      {
         project.setDescription(project.getId());
      }
      compInfo.setDescription( project.getDescription() );

      if (scm != null && scm.getConnection() != null)
      {
         String scmConnStr = scm.getConnection().replaceFirst("scm:", "");
         compInfo.setScm("scm=\"" + scmConnStr + "\"");
      }

      // Build list of artifacts, copy them to the lib directory, and add them to component-info.xml
      HashSet artifactSet = new HashSet();
      Artifact projectArtifact = project.getArtifact();
      if (projectArtifact != null && projectArtifact.getFile() != null)
      {
         String deployFileName = generateOutputFileName(projectArtifact, removeArtifactVersion);
         artifactSet.add("    <artifact id=\"" + deployFileName + "\"/>\n");
         try
         {
            this.getLog().debug("Deploying file://" + componentLibDir + fileSep + deployFileName);
            FileUtils.copyFile(projectArtifact.getFile(), new File(componentLibDir, deployFileName));
         }
         catch (IOException ioe)
         {
            throw new MojoExecutionException("Problem copying artifact: " + ioe.getMessage());
         }
      }

      // Add the attached (non-default) artifacts
      List list = project.getAttachedArtifacts();
      this.getLog().debug("Number of attached artifacts: " + list.size());
      
      Iterator artifactIter = list.iterator();
      while (artifactIter.hasNext())
      {
         Artifact artifact = (Artifact) artifactIter.next();
         this.getLog().debug("Attaching artifact: " + artifact.getArtifactId());
         
         String classifierStr = artifact.getClassifier();
         if (classifierStr == null) 
         {
            classifierStr = "";
            // If the classifier is null and the artifactId matches the project artifactId
            // it must be a composite project, so we add the artifact to the
            // list of exports.
            if (artifact.getArtifactId().equals(project.getArtifactId())) {
               exports.add(artifactId + "." + artifact.getType());
            }
            if (project.getArtifact() != null) {
               getLog().warn("Attached artifact with same classifier as default artifact." + "\n"
                     + "Attached artifact: " + artifact.getFile() + "\n"
                     + "Project artifact: " + project.getArtifact().getFile() + "\n"
                     + "This could be caused by an assembly descriptor with no id.");
            }
         }
         String componentArtifactFileName = generateOutputFileName(artifact, removeArtifactVersion);
         
         // Check for duplicates
         if (compInfo.containsArtifactId( componentArtifactFileName ) )
         {
            this.getLog().warn("Duplicate attached artifact found: " + componentArtifactFileName);
         }
         
         compInfo.addArtifactId( componentArtifactFileName );
         
         try
         {
            this.getLog().debug("Deploying file://" + componentLibDir + fileSep + componentArtifactFileName);
            File componentArtifactFile = new File(componentLibDir, componentArtifactFileName);
            FileUtils.copyFile(artifact.getFile(), componentArtifactFile);
         }
         catch (IOException ioe)
         {
            throw new MojoExecutionException("Problem copying artifact: " + ioe.getMessage());
         }
      }

      // Create and set list of includes for export
      StringBuffer exportsString = new StringBuffer();
      if (project.getArtifact().getFile() != null)
      {
         compInfo.addExport( generateOutputFileName( project.getArtifact(), removeArtifactVersion ) );
      }

      // Generate the list of imports
      addImports( compInfo );

      // Write the component info file
      File compInfoFile = new File( componentVersionDir, ComponentInfoWriter.DEFAULT_COMP_INFO_FILENAME );
      try
      {
         if (!componentVersionDir.exists())
         {
            componentVersionDir.mkdirs();
         }
         this.getLog().info("Deploying file: " + compInfoFile.getCanonicalPath());
         ComponentInfoWriter.loadTemplate();
         ComponentInfoWriter.writeComponentInfo( compInfo, compInfoFile );
      }
      catch (IOException ioe)
      {
         this.getLog().error(ioe);
      }

      ComponentInfoReader verifier = new ComponentInfoReader();
      try
      {
         verifier.verifyCompInfo(new FileInputStream(compInfoFile));
      }
      catch (SAXException e)
      {
         this.getLog().warn(e);
      }
      catch (FileNotFoundException fnfe)
      {
         fnfe.printStackTrace();
      }
   }

   private String generateOutputFileName(Artifact artifact, boolean removeArtifactVersion)
   {
      String deployFileName = artifact.getFile().getName();
      if (removeArtifactVersion)
      {
         if ( deployFileName.contains( "-" + artifact.getBaseVersion() ) )
         {
            deployFileName = deployFileName.replace( ("-" + artifact.getBaseVersion()), "" );
         }
      }
      // Check if the artifactId was customized
      if ( ! project.getArtifactId().equals(artifactId))
      {
         if ( deployFileName.contains( project.getArtifactId() ) )
         {
            deployFileName = deployFileName.replace(project.getArtifactId(), artifactId);
         }
      }
      return deployFileName;
   }
   
   /** 
    * Creates the list of imports to include in the component-info.xml
    */
   private void addImports( ComponentInfo compInfo )
   {
      StringBuffer importsString = new StringBuffer();
      if (imports != null)
      {
         Set componentNames = imports.keySet();
         Iterator compNameIter = componentNames.iterator();
         while (compNameIter.hasNext())
         {
            String componentId = (String)compNameIter.next();
            String compatibleVersions = (String)imports.get(componentId);
            compInfo.addImport(componentId, compatibleVersions);
         }
      }
      if (this.generateImports)
      {
         List dependencies = project.getDependencies();
         Iterator depIter = dependencies.iterator();
         while (depIter.hasNext())
         {
            Dependency dep = (Dependency) depIter.next();
            if (dep.getScope() == null || !dep.getScope().equals("test")) 
            {               
               String componentId = dep.getGroupId().replace('.', '/');
               String compatibleVersions = dep.getVersion();
               compInfo.addImport(componentId, compatibleVersions);
            }
         }
      }
   }

}
