/*
 * Copyright (c) 2009 by Red Hat Inc and/or its affiliates or by
 * third-party contributors as indicated by either @author tags or express
 * copyright attribution statements applied by the authors.  All
 * third-party contributions are distributed under license by Red Hat Inc.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.jboss.maven.plugins.injection;

import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.net.URL;
import java.net.MalformedURLException;
import java.net.URLClassLoader;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.LoaderClassPath;
import javassist.CtMethod;
import javassist.bytecode.ClassFile;
import javassist.bytecode.ConstantAttribute;
import javassist.bytecode.FieldInfo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.artifact.DependencyResolutionRequiredException;

/**
 * Used to inject resolved expression values into compiled bytecode.
 * <p/>
 * TODO : add checks as to whether the injection is needed to avoid file timestamp changes.
 * Basically we should skip the injection if the class file field is already the injection value...
 * 
 * @goal bytecode
 * @phase compile
 * @requiresDependencyResolution
 *
 * @author Steve Ebersole
 */
public class BytecodeInjectionMojo extends AbstractMojo {
	/**
     * The Maven Project Object
     *
     * @parameter expression="${project}"
     * @required
     */
    protected MavenProject project;

	/**
	 * The injections to be performed.
	 *
	 * @parameter
	 * @required
	 */
	protected BytecodeInjection[] bytecodeInjections;

	private LoaderClassPath loaderClassPath;
	private ClassPool classPool;

	/**
	 * {@inheritDoc}
	 */
	public void execute() throws MojoExecutionException, MojoFailureException {
		final ClassLoader mavenProjectCompileClassPathClassLoader = buildProjectCompileClassLoader();
		loaderClassPath = new LoaderClassPath( mavenProjectCompileClassPathClassLoader );

		classPool = new ClassPool( true );
		classPool.appendClassPath( loaderClassPath );

		for ( BytecodeInjection injection : bytecodeInjections ) {
			for ( TargetMember member : injection.getTargetMembers() ) {
				final InjectionTarget injectionTarget;
				if ( member instanceof Constant ) {
					injectionTarget = new ConstantInjectionTarget( ( Constant ) member );
				}
				else if ( member instanceof MethodBodyReturn ) {
					injectionTarget = new MethodBodyReturnReplacementTarget( ( MethodBodyReturn ) member );
				}
				else {
					throw new MojoExecutionException( "Unexpected injection member type : " + member );
				}
				injectionTarget.inject( injection.getExpression() );
			}
		}

		loaderClassPath.close();
	}

	/**
	 * Builds a {@link ClassLoader} based on the maven project's compile classpath elements.
	 *
	 * @return The {@link ClassLoader} made up of the maven project's compile classpath elements.
	 *
	 * @throws MojoExecutionException Indicates an issue processing one of the classpath elements
	 */
	private ClassLoader buildProjectCompileClassLoader() throws MojoExecutionException {
		ArrayList<URL> classPathUrls = new ArrayList<URL>();
		for ( String path : projectCompileClasspathElements() ) {
			try {
				getLog().debug( "Adding project compile classpath element : " + path );
				classPathUrls.add( new File( path ).toURI().toURL() );
			}
			catch ( MalformedURLException e ) {
				throw new MojoExecutionException( "Unable to build path URL [" + path + "]" );
			}
		}
		return new URLClassLoader( classPathUrls.toArray( new URL[classPathUrls.size()] ), getClass().getClassLoader() );
	}

	/**
	 * Essentially a call to {@link MavenProject#getCompileClasspathElements} except that here we
	 * cast it to the generic type and internally handle {@link DependencyResolutionRequiredException}.
	 *
	 * @return The compile classpath elements
	 *
	 * @throws MojoExecutionException Indicates a {@link DependencyResolutionRequiredException} was encountered
	 */
	@SuppressWarnings({ "unchecked" })
	private List<String> projectCompileClasspathElements() throws MojoExecutionException {
		try {
			return ( List<String> ) project.getCompileClasspathElements();
		}
		catch ( DependencyResolutionRequiredException e ) {
			throw new MojoExecutionException(
					"Call to MavenProject#getCompileClasspathElements required dependency resolution"
			);
		}
	}

	/**
	 * Defines a strategy for applying injections.
	 *
	 * @author Steve Ebersole
	 */
	public static interface InjectionTarget {
		/**
		 * Inject the given value per this target's strategy.
		 *
		 * @param value The value to inject.
		 *
		 * @throws MojoExecutionException Indicates a problem performing the injection.
		 */
		public void inject(String value) throws MojoExecutionException;
	}

	private abstract class BaseInjectionTarget implements InjectionTarget {
		private final TargetMember targetMember;

		public TargetMember getTargetMember() {
			return targetMember;
		}

		private final File classFileLocation;

		public File getClassFileLocation() {
			return classFileLocation;
		}

		private final CtClass ctClass;

		public CtClass getCtClass() {
			return ctClass;
		}

		protected BaseInjectionTarget(TargetMember targetMember) throws MojoExecutionException {
			this.targetMember = targetMember;
			try {
				classFileLocation = new File( loaderClassPath.find( targetMember.getClassName() ).toURI() );
				ctClass = classPool.get( targetMember.getClassName() );
			}
			catch ( Throwable e ) {
				throw new MojoExecutionException( "Unable to resolve class file path", e );
			}
		}

		protected void writeOutChanges() throws MojoExecutionException {
			getLog().info( "writing injection changes back [" + classFileLocation.getAbsolutePath() + "]" );
			ClassFile classFile = ctClass.getClassFile();
			classFile.compact();
			try {
				DataOutputStream out = new DataOutputStream( new BufferedOutputStream( new FileOutputStream( classFileLocation ) ) );
				try {

					classFile.write( out );
					out.flush();
					if ( ! classFileLocation.setLastModified( System.currentTimeMillis() ) ) {
						getLog().info( "Unable to manually update class file timestamp" );
					}
				}
				finally {
					out.close();
				}
			}
			catch ( IOException e ) {
				throw new MojoExecutionException( "Unable to write out modified class file", e );
			}
		}
	}

	private class ConstantInjectionTarget extends BaseInjectionTarget {
		private final FieldInfo ctFieldInfo;

		private ConstantInjectionTarget(Constant constant) throws MojoExecutionException {
			super( constant );

			try {
				ctFieldInfo = getCtClass().getField( constant.getFieldName() ).getFieldInfo();
			}
			catch ( Throwable e ) {
				throw new MojoExecutionException( "Unable to resolve class field [" + constant.getQualifiedName() + "]", e );
			}
		}

		public void inject(String value) throws MojoExecutionException {
			ctFieldInfo.addAttribute(
					new ConstantAttribute(
							ctFieldInfo.getConstPool(),
							ctFieldInfo.getConstPool().addStringInfo( value )
					)
			);

			writeOutChanges();
		}
	}

	private class MethodBodyReturnReplacementTarget extends BaseInjectionTarget {
		private final CtMethod ctMethod;

		private MethodBodyReturnReplacementTarget(MethodBodyReturn method) throws MojoExecutionException {
			super( method );

			try {
				for ( CtMethod ctMethod : getCtClass().getMethods() ) {
					if ( method.getMethodName().equals( ctMethod.getName() ) ) {
						this.ctMethod = ctMethod;
						return;
					}
				}
				throw new MojoExecutionException( "Could not locate method [" + method.getQualifiedName() + "]" );
			}
			catch ( Throwable e ) {
				throw new MojoExecutionException( "Unable to resolve method [" + method.getQualifiedName() + "]", e );
			}
		}

		public void inject(String value) throws MojoExecutionException {
			try {
				ctMethod.setBody( "{return \"" + value + "\";}" );
			}
			catch ( Throwable t ) {
				throw new MojoExecutionException( "Unable to replace method body", t );
			}
			writeOutChanges();
		}
	}
}
