/*
 * jDocBook, processing of DocBook sources as a Maven plugin
 *
 * Copyright (c) 2008, Red Hat Middleware LLC or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Middleware LLC.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 *
 */
package org.jboss.jdocbook.i18n.gettext;

import java.io.File;
import java.util.Locale;

import org.jboss.jdocbook.JDocBookProcessException;
import org.jboss.jdocbook.Options;
import org.jboss.jdocbook.i18n.PoSynchronizer;
import org.jboss.jdocbook.util.FileUtils;
import org.jboss.jdocbook.util.I18nUtils;
import org.jboss.jdocbook.util.VCSDirectoryExclusionFilter;
import org.jboss.maven.shared.process.Executor;

/**
 * Implementation of the {@link PoSynchronizer} contract based on system calls
 * to either the 'msgmerge' or the 'msginit' commands (both part of the GNU
 * gettext package).
 *
 * @author Steve Ebersole
 */
public class PoSynchronizerImpl implements PoSynchronizer {
	/**
	 * {@inheritDoc}
	 */
	public void synchronizePo(File potDirectory, File translationDirectory, Locale translationLocale, Options options)
			throws JDocBookProcessException {
		if ( !potDirectory.exists() ) {
			options.getLog().info( "skipping PO updates; POT directory did not exist : {0}", potDirectory );
			return;
		}
		File[] files = potDirectory.listFiles( new VCSDirectoryExclusionFilter() );
		for ( int i = 0, X = files.length; i < X; i++) {
			if ( files[i].isDirectory() ) {
				// recurse into the directory by calling back into ourselves with the sub-dir
				synchronizePo(
						new File( potDirectory, files[i].getName() ),
						new File( translationDirectory, files[i].getName() ),
						translationLocale,
						options
				);
			}
			else {
				if ( I18nUtils.isPotFile( files[i] ) ) {
					File translation = new File( translationDirectory, I18nUtils.determinePoFileName( files[i] ) );
					updateTranslation( files[i], translation, translationLocale, options );
				}
			}
		}
	}

	private void updateTranslation(File template, File translation, Locale translationLocale, Options options) {
		if ( !template.exists() ) {
			options.getLog().trace( "skipping PO updates; POT file did not exist : {0}", template );
			return;
		}

		if ( translation.lastModified() >= template.lastModified() ) {
			options.getLog().trace( "skipping PO updates; up-to-date : {0}", translation );
			return;
		}

		final String cmd;
		if ( translation.exists() ) {
			cmd = "msgmerge --quiet --backup=none --update " + FileUtils.resolveFullPathName( translation )
					+ " " + FileUtils.resolveFullPathName( template );
		}
		else {
			translation.getParentFile().mkdirs();
			cmd = "msginit --no-translator -l " + translationLocale
					+ " -i " + FileUtils.resolveFullPathName( template )
					+ " -o " + FileUtils.resolveFullPathName( translation );
		}
		Executor.execute( cmd );
	}
}
