/*
 * jDocBook, processing of DocBook sources as a Maven plugin
 *
 * Copyright (c) 2008, Red Hat Middleware LLC or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Middleware LLC.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 *
 */

package org.jboss.jdocbook.i18n.gettext;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import org.jboss.jdocbook.JDocBookProcessException;
import org.jboss.jdocbook.Options;
import org.jboss.jdocbook.i18n.PotSynchronizer;
import org.jboss.jdocbook.util.FileUtils;
import org.jboss.jdocbook.util.I18nUtils;
import org.jboss.jdocbook.util.XIncludeHelper;
import org.jboss.maven.shared.process.Executor;

/**
 * Implementation of the {@link PotSynchronizer} contract based on system calls
 * to the xml2pot command (part of the KDE poxml package).
 *
 * @author Steve Ebersole
 */
public class PotSynchronizerImpl implements PotSynchronizer {
	/**
	 * {@inheritDoc} 
	 */
	public void synchronizePot(File masterFile, File templateDirectory, Options options) throws JDocBookProcessException {
		if ( !masterFile.exists() ) {
			options.getLog().info( "skipping POT updates; master file did not exist : {0}", masterFile );
			return;
		}
		final File sourceBasedir = masterFile.getParentFile();
		final String potFileName = I18nUtils.determinePotFileName( masterFile );
		final File potFile = new File( templateDirectory, potFileName );
		updatePortableObjectTemplate( masterFile, potFile, options );

		// Note : recursion below accounts for inclusions within inclusions
		for ( File inclusion : XIncludeHelper.locateInclusions( masterFile ) ) {
			final String relativity = FileUtils.determineRelativity( inclusion, sourceBasedir );
			final File relativeTemplateDir = ( relativity == null ) ? templateDirectory : new File(
					templateDirectory,
					relativity
			);
			synchronizePot( inclusion, relativeTemplateDir, options );
		}
	}

	private void updatePortableObjectTemplate(File masterFile, File potFile, Options options) {
		if ( !masterFile.exists() ) {
			options.getLog().trace( "skipping POT update; source file did not exist : {0}", masterFile );
			return;
		}

		if ( potFile.exists() && potFile.lastModified() >= masterFile.lastModified() ) {
			options.getLog().trace( "skipping POT update; up-to-date : {0}", potFile );
			return;
		}

		potFile.getParentFile().mkdirs();
		executeXml2pot( masterFile, potFile, options );
	}

	private void executeXml2pot(File masterFile, File potFile, Options options) {
		final String cmd = "xml2pot " + FileUtils.resolveFullPathName( masterFile );

		try {
			final FileOutputStream xmlStream = new FileOutputStream( potFile );
			try {
				options.getLog().trace( "updating POT file {0}", potFile );
				Executor.execute( cmd, xmlStream );
			}
			finally {
				try {
					xmlStream.flush();
					xmlStream.close();
				}
				catch ( IOException ignore ) {
					// intentionally empty...
				}
			}
		}
		catch ( IOException e  ) {
			throw new JDocBookProcessException( "unable to open output stream for POT file [" + potFile + "]" );
		}
	}
}
